<?php

namespace Tests\Feature\Api;

use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ApiTokenAbilityTest extends TestCase
{
    use RefreshDatabase;

    public function test_read_only_task_token_cannot_update_task_status(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
        ]);

        $token = $user->createToken('read-only', ['tasks.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/tasks')
            ->assertOk();

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/tasks/'.$task->id.'/status', [
                'status' => 'in_progress',
            ])
            ->assertForbidden();

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'status' => 'todo',
        ]);
    }

    public function test_update_task_token_can_update_task_status(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
        ]);

        $token = $user->createToken('tasks-update', ['tasks.update'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/tasks/'.$task->id.'/status', [
                'status' => 'review',
            ])
            ->assertOk()
            ->assertJsonPath('data.id', $task->id)
            ->assertJsonPath('data.status', 'review');

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'status' => 'review',
        ]);
    }
}
