<?php

namespace Tests\Concerns;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use ZipArchive;

trait BuildsUpdatePackages
{
    private function createUpdateTargetFixture(): string
    {
        $targetPath = storage_path('framework/testing/update-target-'.bin2hex(random_bytes(5)));

        File::ensureDirectoryExists($targetPath.'/config');
        File::ensureDirectoryExists($targetPath.'/resources');
        File::put($targetPath.'/artisan', "#!/usr/bin/env php\n<?php echo 'artisan';\n");
        File::put($targetPath.'/config/app.php', "<?php\n\nreturn ['name' => 'CRM25'];\n");
        File::put($targetPath.'/resources/obsolete.txt', 'old');

        return $targetPath;
    }

    /**
     * @param  array<string, string>  $files
     * @return array{path:string,checksum:string}
     */
    private function createUpdatePackageFixture(array $files): array
    {
        $packagePath = storage_path('framework/testing/update-package-'.bin2hex(random_bytes(5)).'.zip');
        File::ensureDirectoryExists(dirname($packagePath));

        $zip = new ZipArchive();
        if ($zip->open($packagePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new \RuntimeException('Unable to create update package archive.');
        }

        foreach ($files as $relativePath => $contents) {
            $directory = dirname($relativePath);
            if ($directory !== '.' && $directory !== '') {
                $zip->addEmptyDir($directory);
            }

            $zip->addFromString($relativePath, $contents);
        }

        $zip->close();

        return [
            'path' => $packagePath,
            'checksum' => hash_file('sha256', $packagePath) ?: '',
        ];
    }

    private function fakeAvailableUpdate(string $packagePath, string $checksum, string $version = '1.002', string $build = '1002'): void
    {
        Http::fake([
            'https://update.crm25.webnet.kz/api/v1/crm25/updates*' => Http::response([
                'version' => $version,
                'build' => $build,
                'is_update_available' => true,
                'notes' => 'Release available.',
                'download_url' => 'https://update.crm25.webnet.kz/downloads/2/crm25-'.$version.'.zip',
                'checksum_sha256' => $checksum,
                'file_name' => 'crm25-'.$version.'.zip',
            ], 200),
            'https://update.crm25.webnet.kz/downloads/2/*' => Http::response(file_get_contents($packagePath) ?: '', 200, [
                'Content-Type' => 'application/zip',
            ]),
        ]);
    }
}
