<?php

namespace Database\Seeders;

use App\Models\DealStage;
use App\Models\OrganizationSetting;
use App\Models\Pipeline;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class ProductionBootstrapSeeder extends Seeder
{
    public function run(): void
    {
        OrganizationSetting::current();

        $host = $this->resolveHost();
        $adminEmail = $this->resolveAdminEmail($host);
        $adminPassword = $this->resolveAdminPassword();
        $adminName = trim((string) env('CRM_BOOTSTRAP_ADMIN_NAME', 'CRM Admin')) ?: 'CRM Admin';

        $admin = User::query()->firstOrNew([
            'email' => $adminEmail,
        ]);

        $admin->fill([
            'name' => $adminName,
            'phone' => trim((string) env('CRM_BOOTSTRAP_ADMIN_PHONE', '+7-700-000-0000')) ?: '+7-700-000-0000',
            'job_title' => trim((string) env('CRM_BOOTSTRAP_ADMIN_JOB_TITLE', 'CRM Administrator')) ?: 'CRM Administrator',
            'role' => 'admin',
            'locale' => trim((string) env('CRM_BOOTSTRAP_ADMIN_LOCALE', 'ru')) ?: 'ru',
            'email_verified_at' => now(),
            'permissions' => [],
        ]);

        if (! $admin->exists) {
            $admin->password = Hash::make($adminPassword);
        }

        $admin->save();

        $pipeline = Pipeline::query()->firstOrCreate(
            [
                'name' => trim((string) env('CRM_BOOTSTRAP_PIPELINE_NAME', 'Primary Sales')) ?: 'Primary Sales',
            ],
            [
                'description' => trim((string) env('CRM_BOOTSTRAP_PIPELINE_DESCRIPTION', 'Default sales pipeline created during server installation.'))
                    ?: 'Default sales pipeline created during server installation.',
                'creator_id' => $admin->id,
                'is_default' => true,
                'is_active' => true,
                'sort_order' => 0,
            ]
        );

        Pipeline::query()
            ->whereKeyNot($pipeline->id)
            ->where('is_default', true)
            ->update(['is_default' => false]);

        $pipeline->forceFill([
            'creator_id' => $pipeline->creator_id ?: $admin->id,
            'is_default' => true,
            'is_active' => true,
            'sort_order' => 0,
        ])->save();

        $stageDefinitions = [
            ['code' => 'new_lead', 'name' => 'New Lead', 'probability' => 10, 'color' => '#94a3b8', 'is_won' => false, 'is_lost' => false],
            ['code' => 'qualification', 'name' => 'Qualification', 'probability' => 30, 'color' => '#3b82f6', 'is_won' => false, 'is_lost' => false],
            ['code' => 'proposal', 'name' => 'Proposal', 'probability' => 60, 'color' => '#8b5cf6', 'is_won' => false, 'is_lost' => false],
            ['code' => 'negotiation', 'name' => 'Negotiation', 'probability' => 80, 'color' => '#f59e0b', 'is_won' => false, 'is_lost' => false],
            ['code' => 'won', 'name' => 'Won', 'probability' => 100, 'color' => '#10b981', 'is_won' => true, 'is_lost' => false],
            ['code' => 'lost', 'name' => 'Lost', 'probability' => 0, 'color' => '#ef4444', 'is_won' => false, 'is_lost' => true],
        ];

        foreach ($stageDefinitions as $index => $stage) {
            DealStage::query()->updateOrCreate(
                [
                    'pipeline_id' => $pipeline->id,
                    'code' => $stage['code'],
                ],
                [
                    'name' => $stage['name'],
                    'sort_order' => $index,
                    'probability' => $stage['probability'],
                    'color' => $stage['color'],
                    'is_won' => $stage['is_won'],
                    'is_lost' => $stage['is_lost'],
                ]
            );
        }
    }

    private function resolveHost(): string
    {
        $appUrl = trim((string) env('APP_URL', ''));
        $host = is_string(parse_url($appUrl, PHP_URL_HOST)) ? (string) parse_url($appUrl, PHP_URL_HOST) : '';
        $host = trim(Str::lower($host));

        return $host !== '' ? $host : 'crm.local';
    }

    private function resolveAdminEmail(string $host): string
    {
        $explicit = trim((string) env('CRM_BOOTSTRAP_ADMIN_EMAIL', ''));
        if ($explicit !== '') {
            return Str::lower($explicit);
        }

        if (filter_var($host, FILTER_VALIDATE_IP)) {
            return 'admin@crm.local';
        }

        return 'admin@'.Str::lower($host);
    }

    private function resolveAdminPassword(): string
    {
        $password = trim((string) env('CRM_BOOTSTRAP_ADMIN_PASSWORD', ''));

        return $password !== '' ? $password : 'password';
    }
}
