<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (DB::getDriverName() === 'sqlite') {
            return;
        }

        if (Schema::hasTable('deal_stages') && Schema::hasTable('pipelines') && ! $this->foreignKeyExists('deal_stages', 'deal_stages_pipeline_id_foreign')) {
            Schema::table('deal_stages', function (Blueprint $table) {
                $table->foreign('pipeline_id', 'deal_stages_pipeline_id_foreign')
                    ->references('id')
                    ->on('pipelines')
                    ->cascadeOnDelete();
            });
        }

        if (Schema::hasTable('deals') && Schema::hasTable('pipelines') && ! $this->foreignKeyExists('deals', 'deals_pipeline_id_foreign')) {
            Schema::table('deals', function (Blueprint $table) {
                $table->foreign('pipeline_id', 'deals_pipeline_id_foreign')
                    ->references('id')
                    ->on('pipelines');
            });
        }

        if (Schema::hasTable('deals') && Schema::hasTable('deal_stages') && ! $this->foreignKeyExists('deals', 'deals_stage_id_foreign')) {
            Schema::table('deals', function (Blueprint $table) {
                $table->foreign('stage_id', 'deals_stage_id_foreign')
                    ->references('id')
                    ->on('deal_stages');
            });
        }
    }

    public function down(): void
    {
        if (DB::getDriverName() === 'sqlite') {
            return;
        }

        if (Schema::hasTable('deals') && $this->foreignKeyExists('deals', 'deals_stage_id_foreign')) {
            Schema::table('deals', function (Blueprint $table) {
                $table->dropForeign('deals_stage_id_foreign');
            });
        }

        if (Schema::hasTable('deals') && $this->foreignKeyExists('deals', 'deals_pipeline_id_foreign')) {
            Schema::table('deals', function (Blueprint $table) {
                $table->dropForeign('deals_pipeline_id_foreign');
            });
        }

        if (Schema::hasTable('deal_stages') && $this->foreignKeyExists('deal_stages', 'deal_stages_pipeline_id_foreign')) {
            Schema::table('deal_stages', function (Blueprint $table) {
                $table->dropForeign('deal_stages_pipeline_id_foreign');
            });
        }
    }

    private function foreignKeyExists(string $table, string $constraint): bool
    {
        if (DB::getDriverName() !== 'mysql') {
            return false;
        }

        return (bool) DB::table('information_schema.TABLE_CONSTRAINTS')
            ->whereRaw('TABLE_SCHEMA = DATABASE()')
            ->where('TABLE_NAME', $table)
            ->where('CONSTRAINT_NAME', $constraint)
            ->where('CONSTRAINT_TYPE', 'FOREIGN KEY')
            ->exists();
    }
};
