#!/usr/bin/env bash
set -Eeuo pipefail

GUAC_VERSION="${GUAC_VERSION:-1.6.0}"
TOMCAT_VERSION="${TOMCAT_VERSION:-9.0.115}"
GATEWAY_DIR="${GATEWAY_DIR:-/var/www/update.crm25.webnet.kz/remote-browser-gateway}"
PUBLIC_ROOT="${PUBLIC_ROOT:-/var/www/update.crm25.webnet.kz/public}"
SITE_CONF="${SITE_CONF:-/etc/nginx/sites-available/update.crm25.webnet.kz}"
GUAC_HOME="${GUAC_HOME:-/etc/guacamole}"
REMOTE_BROWSER_GATEWAY_SECRET="${REMOTE_BROWSER_GATEWAY_SECRET:-}"
CATALINA_DIR="${CATALINA_DIR:-${GATEWAY_DIR}/tomcat}"
TOMCAT_USER="${TOMCAT_USER:-tomcat}"
TOMCAT_PORT="${TOMCAT_PORT:-8090}"

if [[ -z "${REMOTE_BROWSER_GATEWAY_SECRET}" || ! "${REMOTE_BROWSER_GATEWAY_SECRET}" =~ ^[0-9a-fA-F]{32}$ ]]; then
  echo "REMOTE_BROWSER_GATEWAY_SECRET must be a 32-character hex string." >&2
  exit 1
fi

install -d -m 0755 "${GATEWAY_DIR}"/{bin,launcher,downloads,logs}
install -d -m 0755 "${GUAC_HOME}"/{extensions,lib}

export DEBIAN_FRONTEND=noninteractive
apt-get update
apt-get install -y \
  curl \
  ca-certificates \
  openjdk-21-jre-headless \
  guacd \
  php8.4-cli \
  php8.4-curl

if ! id -u "${TOMCAT_USER}" >/dev/null 2>&1; then
  useradd --system --home-dir "${GATEWAY_DIR}" --shell /usr/sbin/nologin "${TOMCAT_USER}"
fi

curl -fsSL "https://downloads.apache.org/guacamole/${GUAC_VERSION}/binary/guacamole-${GUAC_VERSION}.war" -o "${GATEWAY_DIR}/downloads/guacamole-${GUAC_VERSION}.war"
curl -fsSL "https://downloads.apache.org/guacamole/${GUAC_VERSION}/binary/guacamole-auth-json-${GUAC_VERSION}.tar.gz" -o "${GATEWAY_DIR}/downloads/guacamole-auth-json-${GUAC_VERSION}.tar.gz"
curl -fsSL "https://downloads.apache.org/tomcat/tomcat-9/v${TOMCAT_VERSION}/bin/apache-tomcat-${TOMCAT_VERSION}.tar.gz" -o "${GATEWAY_DIR}/downloads/apache-tomcat-${TOMCAT_VERSION}.tar.gz"

tmp_dir="$(mktemp -d)"
trap 'rm -rf "${tmp_dir}"' EXIT
tar -xzf "${GATEWAY_DIR}/downloads/guacamole-auth-json-${GUAC_VERSION}.tar.gz" -C "${tmp_dir}"
rm -rf "${CATALINA_DIR}"
mkdir -p "${CATALINA_DIR}"
tar -xzf "${GATEWAY_DIR}/downloads/apache-tomcat-${TOMCAT_VERSION}.tar.gz" -C "${tmp_dir}"
cp -a "${tmp_dir}/apache-tomcat-${TOMCAT_VERSION}/." "${CATALINA_DIR}/"
rm -rf "${CATALINA_DIR}/webapps/"*
sed -i "s/port=\"8080\"/port=\"${TOMCAT_PORT}\"/g" "${CATALINA_DIR}/conf/server.xml"
sed -i 's/port="8005"/port="8006"/g' "${CATALINA_DIR}/conf/server.xml"

install -m 0644 "${GATEWAY_DIR}/downloads/guacamole-${GUAC_VERSION}.war" "${CATALINA_DIR}/webapps/remote-browser.war"
install -m 0644 "${tmp_dir}/guacamole-auth-json-${GUAC_VERSION}/guacamole-auth-json-${GUAC_VERSION}.jar" "${GUAC_HOME}/extensions/"

cat > "${GUAC_HOME}/guacamole.properties" <<EOF
guacd-hostname: 127.0.0.1
guacd-port: 4822
extension-priority: auth-json
json-secret-key: ${REMOTE_BROWSER_GATEWAY_SECRET}
EOF

cat > "${CATALINA_DIR}/bin/setenv.sh" <<EOF
#!/usr/bin/env bash
export GUACAMOLE_HOME="${GUAC_HOME}"
export JAVA_HOME="/usr/lib/jvm/java-21-openjdk-amd64"
EOF
chmod +x "${CATALINA_DIR}/bin/setenv.sh"
chown -R "${TOMCAT_USER}:${TOMCAT_USER}" "${CATALINA_DIR}"

rm -rf "${PUBLIC_ROOT}/remote-browser-launch"
ln -s "${GATEWAY_DIR}/launcher" "${PUBLIC_ROOT}/remote-browser-launch"

python3 - <<'PY' "${SITE_CONF}"
from pathlib import Path
import re
import sys

path = Path(sys.argv[1])
text = path.read_text()

launch_block = """
    location /remote-browser-launch/ {
        try_files $uri $uri/ /remote-browser-launch/index.php?$query_string;
    }

    location ^~ /remote-browser/ {
        proxy_pass http://127.0.0.1:8090/remote-browser/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Forwarded-Host $host;
        proxy_set_header X-Forwarded-Proto https;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection "upgrade";
        proxy_cookie_path / /remote-browser/;
        proxy_buffering off;
        proxy_read_timeout 3600s;
        proxy_send_timeout 3600s;
    }
"""
text = re.sub(r"\n\s*location\s+\^?\~?\s*/remote-browser-launch/\s*\{.*?\n\s*\}\n", "\n", text, flags=re.S)
text = re.sub(r"\n\s*location\s+\^?\~?\s*/remote-browser/\s*\{.*?\n\s*\}\n", "\n", text, flags=re.S)

marker = "    location / {\n        try_files $uri /index.php?$query_string;\n    }\n"
if marker not in text:
    raise SystemExit("Unable to patch nginx config: marker not found")

text = text.replace(marker, marker + launch_block + "\n")
path.write_text(text)
PY

cat > /etc/systemd/system/crm25-remote-browser.service <<EOF
[Unit]
Description=CRM25 Browser Remote Gateway
After=network.target guacd.service
Requires=guacd.service

[Service]
Type=simple
User=${TOMCAT_USER}
Group=${TOMCAT_USER}
Environment=JAVA_HOME=/usr/lib/jvm/java-21-openjdk-amd64
Environment=CATALINA_HOME=${CATALINA_DIR}
Environment=CATALINA_BASE=${CATALINA_DIR}
Environment=GUACAMOLE_HOME=${GUAC_HOME}
WorkingDirectory=${CATALINA_DIR}
ExecStart=${CATALINA_DIR}/bin/catalina.sh run
Restart=on-failure
RestartSec=3

[Install]
WantedBy=multi-user.target
EOF

nginx -t
systemctl daemon-reload
systemctl enable guacd crm25-remote-browser
systemctl restart guacd crm25-remote-browser php8.4-fpm nginx

echo "Browser gateway installed."
echo "Guacamole app: https://update.crm25.webnet.kz/remote-browser/"
echo "Launcher URL: https://update.crm25.webnet.kz/remote-browser-launch/"
