<?php

namespace App\Support;

use App\Models\UpdateCheck;
use App\Models\UpdateSetting;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Schema;
use RuntimeException;

class UpdateCenterManager
{
    public const DEFAULT_FEED_URL = 'https://update.crm25.webnet.kz';

    public const DEFAULT_CHANNEL = 'stable';

    private ?bool $storageReady = null;

    public function storageIsReady(): bool
    {
        if ($this->storageReady !== null) {
            return $this->storageReady;
        }

        $this->storageReady = Schema::hasTable('update_settings')
            && Schema::hasTable('update_checks');
        $this->storageReady = $this->storageReady && Schema::hasTable('update_installations');

        return $this->storageReady;
    }

    public function settings(): UpdateSetting
    {
        if (! $this->storageIsReady()) {
            return new UpdateSetting($this->defaultSettingsPayload());
        }

        $settings = UpdateSetting::query()->first();

        if (! $settings) {
            $settings = new UpdateSetting($this->defaultSettingsPayload());
            $settings->save();
        }

        if (trim((string) $settings->current_version) === '') {
            $settings->current_version = $this->defaultCurrentVersion();
        }

        if (trim((string) $settings->feed_url) === '') {
            $settings->feed_url = self::DEFAULT_FEED_URL;
        }

        return $settings;
    }

    public function defaultCurrentVersion(): string
    {
        $version = trim((string) config('app.version', ''));

        return $version !== '' ? $version : '1.011';
    }

    public function normalizeFeedUrl(?string $feedUrl): string
    {
        $url = trim((string) $feedUrl);
        if ($url === '') {
            return self::DEFAULT_FEED_URL;
        }

        if (! preg_match('~^https?://~i', $url)) {
            $url = 'https://'.$url;
        }

        $parsed = parse_url($url);
        if (! is_array($parsed) || ! isset($parsed['host']) || trim((string) $parsed['host']) === '') {
            return self::DEFAULT_FEED_URL;
        }

        $scheme = strtolower((string) ($parsed['scheme'] ?? 'https'));
        if (! in_array($scheme, ['http', 'https'], true)) {
            $scheme = 'https';
        }

        $normalized = $scheme.'://'.$parsed['host'];
        if (isset($parsed['port'])) {
            $normalized .= ':'.(int) $parsed['port'];
        }

        $path = trim((string) ($parsed['path'] ?? ''));
        if ($path !== '') {
            $normalized .= '/'.ltrim($path, '/');
        }

        $query = trim((string) ($parsed['query'] ?? ''));
        if ($query !== '') {
            $normalized .= '?'.$query;
        }

        return rtrim($normalized, '/');
    }

    public function checkNow(?UpdateSetting $settings = null): UpdateCheck
    {
        if (! $this->storageIsReady()) {
            return new UpdateCheck([
                'status' => 'failed',
                'is_update_available' => false,
                'error_message' => 'Update tables are not migrated. Run "php artisan migrate".',
                'checked_at' => now(),
            ]);
        }

        $settings ??= $this->settings();

        $feedUrl = $this->normalizeFeedUrl((string) $settings->feed_url);
        $timeout = max(2, min(60, (int) ($settings->request_timeout_seconds ?: 8)));
        $verifyTls = (bool) $settings->verify_tls;
        $currentVersion = trim((string) ($settings->current_version ?: $this->defaultCurrentVersion()));
        $channel = trim((string) ($settings->channel ?: self::DEFAULT_CHANNEL));
        $candidateEndpoints = $this->candidateEndpoints($feedUrl);

        $lastError = 'Update source did not return a valid response.';
        $lastStatus = null;
        $lastEndpoint = null;
        $check = null;

        foreach ($candidateEndpoints as $endpoint) {
            $lastEndpoint = $endpoint;

            try {
                $response = Http::timeout($timeout)
                    ->acceptJson()
                    ->withOptions(['verify' => $verifyTls])
                    ->get($endpoint, [
                        'product' => 'crm25',
                        'channel' => $channel,
                        'version' => $currentVersion,
                        'app_url' => (string) config('app.url', ''),
                    ]);

                $lastStatus = $response->status();

                if (! $response->successful()) {
                    $lastError = sprintf('HTTP %d', $response->status());

                    continue;
                }

                /** @var mixed $payload */
                $payload = $response->json();
                if (! is_array($payload)) {
                    throw new RuntimeException('Invalid JSON response.');
                }

                $remoteVersion = $this->firstNonEmptyString([
                    Arr::get($payload, 'version'),
                    Arr::get($payload, 'latest_version'),
                    Arr::get($payload, 'data.version'),
                    Arr::get($payload, 'release.version'),
                ]);

                $remoteBuild = $this->firstNonEmptyString([
                    Arr::get($payload, 'build'),
                    Arr::get($payload, 'latest_build'),
                    Arr::get($payload, 'data.build'),
                    Arr::get($payload, 'release.build'),
                ]);

                $message = $this->firstNonEmptyString([
                    Arr::get($payload, 'notes'),
                    Arr::get($payload, 'message'),
                    Arr::get($payload, 'summary'),
                    Arr::get($payload, 'release.notes'),
                ]);

                $isAvailable = $this->resolveUpdateAvailability($payload, $remoteVersion, $currentVersion);

                $check = UpdateCheck::query()->create([
                    'update_setting_id' => $settings->id,
                    'status' => 'success',
                    'endpoint_url' => $endpoint,
                    'http_status' => $response->status(),
                    'is_update_available' => $isAvailable,
                    'remote_version' => $remoteVersion,
                    'remote_build' => $remoteBuild,
                    'error_message' => $message,
                    'payload' => $payload,
                    'checked_at' => now(),
                ]);

                $settings->fill([
                    'feed_url' => $feedUrl,
                    'current_version' => $currentVersion,
                    'channel' => $channel,
                    'last_checked_at' => now(),
                    'last_remote_version' => $remoteVersion,
                    'last_remote_build' => $remoteBuild,
                    'last_error' => null,
                ])->save();

                return $check;
            } catch (\Throwable $exception) {
                $lastError = trim($exception->getMessage()) !== ''
                    ? $exception->getMessage()
                    : 'Unknown update check error.';
            }
        }

        $check = UpdateCheck::query()->create([
            'update_setting_id' => $settings->id,
            'status' => 'failed',
            'endpoint_url' => $lastEndpoint,
            'http_status' => $lastStatus,
            'is_update_available' => false,
            'remote_version' => null,
            'remote_build' => null,
            'error_message' => $lastError,
            'payload' => null,
            'checked_at' => now(),
        ]);

        $settings->fill([
            'feed_url' => $feedUrl,
            'current_version' => $currentVersion,
            'channel' => $channel,
            'last_checked_at' => now(),
            'last_error' => $lastError,
        ])->save();

        return $check;
    }

    /**
     * @return list<string>
     */
    private function candidateEndpoints(string $feedUrl): array
    {
        $base = rtrim($feedUrl, '/');

        return array_values(array_unique([
            $base.'/api/v1/crm25/updates',
            $base.'/api/v1/updates',
            $base.'/api/updates',
            $base.'/updates.json',
            $base.'/update.json',
            $base,
        ]));
    }

    /**
     * @param  array<int, mixed>  $candidates
     */
    private function firstNonEmptyString(array $candidates): ?string
    {
        foreach ($candidates as $candidate) {
            $value = trim((string) $candidate);
            if ($value !== '') {
                return $value;
            }
        }

        return null;
    }

    /**
     * @param  array<string, mixed>  $payload
     */
    private function resolveUpdateAvailability(array $payload, ?string $remoteVersion, string $currentVersion): bool
    {
        if (is_bool($payload['is_update_available'] ?? null)) {
            return (bool) $payload['is_update_available'];
        }

        if (is_bool($payload['update_available'] ?? null)) {
            return (bool) $payload['update_available'];
        }

        if ($remoteVersion === null || trim($remoteVersion) === '') {
            return false;
        }

        if ($this->isVersionComparable($currentVersion) && $this->isVersionComparable($remoteVersion)) {
            return version_compare($remoteVersion, $currentVersion, '>');
        }

        return $remoteVersion !== $currentVersion;
    }

    private function isVersionComparable(string $version): bool
    {
        return (bool) preg_match('/^\d+(\.\d+){1,3}([\-+][A-Za-z0-9.\-]+)?$/', trim($version));
    }

    /**
     * @return array<string, mixed>
     */
    private function defaultSettingsPayload(): array
    {
        return [
            'feed_url' => self::DEFAULT_FEED_URL,
            'current_version' => $this->defaultCurrentVersion(),
            'channel' => self::DEFAULT_CHANNEL,
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ];
    }
}
