<?php

namespace App\Support;

use App\Models\RemoteAccessRequest;
use App\Models\User;

class RemoteBrowserGatewayManager
{
    public function isConfigured(): bool
    {
        return config('remote_browser_gateway.enabled')
            && $this->baseUrl() !== ''
            && $this->binarySecret() !== null;
    }

    /**
     * @param  array<string, string>  $snapshot
     */
    public function urlForRequest(User $actor, RemoteAccessRequest $request, array $snapshot): ?string
    {
        if (($snapshot['web_url'] ?? '') !== '') {
            return $snapshot['web_url'];
        }

        if (! $this->isConfigured() || ($snapshot['provider'] ?? '') !== 'rdp') {
            return null;
        }

        $hostname = trim((string) ($snapshot['address'] ?? ''));
        $username = trim((string) ($snapshot['username'] ?? ''));
        $password = trim((string) ($snapshot['password'] ?? ''));
        $port = (string) ($snapshot['port'] ?? config('remote_browser_gateway.rdp.default_port', 3389));

        if ($hostname === '' || $username === '' || $password === '') {
            return null;
        }

        $connectionName = sprintf(
            'Remote access #%d for %s',
            $request->id,
            trim((string) ($request->recipient?->name ?: $request->requester?->name ?: 'CRM user'))
        );

        $payload = [
            'username' => 'crm25-'.$actor->id.'-'.$request->id,
            'expires' => now()->addSeconds((int) config('remote_browser_gateway.ttl_seconds', 300))->getTimestampMs(),
            'connections' => [
                $connectionName => [
                    'protocol' => 'rdp',
                    'parameters' => [
                        'hostname' => $hostname,
                        'port' => $port,
                        'username' => $username,
                        'password' => $password,
                        'ignore-cert' => (string) config('remote_browser_gateway.rdp.ignore_cert', 'true'),
                        'security' => (string) config('remote_browser_gateway.rdp.security', 'any'),
                        'resize-method' => (string) config('remote_browser_gateway.rdp.resize_method', 'display-update'),
                        'enable-wallpaper' => (string) config('remote_browser_gateway.rdp.enable_wallpaper', 'true'),
                        'enable-font-smoothing' => (string) config('remote_browser_gateway.rdp.enable_font_smoothing', 'true'),
                        'enable-desktop-composition' => (string) config('remote_browser_gateway.rdp.enable_desktop_composition', 'true'),
                    ],
                ],
            ],
        ];

        $data = $this->encodePayload($payload);
        if ($data === null) {
            return null;
        }

        $separator = str_contains($this->baseUrl(), '?') ? '&' : '?';

        return $this->baseUrl().$separator.'data='.rawurlencode($data);
    }

    private function baseUrl(): string
    {
        return rtrim(trim((string) config('remote_browser_gateway.base_url', '')), '/').'/';
    }

    private function binarySecret(): ?string
    {
        $secret = trim((string) config('remote_browser_gateway.secret_key', ''));
        if ($secret === '' || strlen($secret) !== 32 || ! ctype_xdigit($secret)) {
            return null;
        }

        $binary = @hex2bin($secret);

        return $binary === false ? null : $binary;
    }

    /**
     * @param  array<string, mixed>  $payload
     */
    private function encodePayload(array $payload): ?string
    {
        $secret = $this->binarySecret();
        if ($secret === null) {
            return null;
        }

        $json = json_encode($payload, JSON_UNESCAPED_SLASHES);
        if ($json === false) {
            return null;
        }

        $signed = hash_hmac('sha256', $json, $secret, true).$json;
        $encrypted = openssl_encrypt(
            $signed,
            'AES-128-CBC',
            $secret,
            OPENSSL_RAW_DATA,
            str_repeat("\0", 16)
        );

        if (! is_string($encrypted) || $encrypted === '') {
            return null;
        }

        return base64_encode($encrypted);
    }
}
