<?php

namespace App\Support;

use App\Models\RemoteAccessRequest;
use App\Models\User;

class RemoteAccessManager
{
    /**
     * @return array<string, string>
     */
    public static function providers(): array
    {
        return [
            'anydesk' => 'AnyDesk',
            'rustdesk' => 'RustDesk',
            'teamviewer' => 'TeamViewer',
            'rdp' => 'RDP',
            'other' => 'Other',
        ];
    }

    public static function hasProfileConnection(User $user): bool
    {
        return self::snapshotForUser($user) !== null;
    }

    /**
     * @return array<string, string>|null
     */
    public static function snapshotForUser(User $user): ?array
    {
        return self::normalizeSnapshot([
            'provider' => $user->remote_access_provider,
            'address' => $user->remote_access_address,
            'username' => $user->remote_access_username,
            'port' => $user->remote_access_port,
            'password' => $user->remote_access_password,
            'launch_url' => $user->remote_access_launch_url,
            'web_url' => $user->remote_access_web_url,
            'notes' => $user->remote_access_notes,
        ]);
    }

    /**
     * @param  array<string, mixed>  $attributes
     * @return array<string, mixed>
     */
    public static function applyProfileDefaults(array $attributes): array
    {
        $provider = trim((string) ($attributes['remote_access_provider'] ?? ''));
        $address = trim((string) ($attributes['remote_access_address'] ?? ''));
        $port = self::normalizePortValue($attributes['remote_access_port'] ?? null, $provider);
        $launchUrl = trim((string) ($attributes['remote_access_launch_url'] ?? ''));
        $webUrl = trim((string) ($attributes['remote_access_web_url'] ?? ''));

        if ($provider === '' || $address === '') {
            return $attributes;
        }

        if ($port !== null) {
            $attributes['remote_access_port'] = $port;
        }

        if ($launchUrl === '') {
            $launchUrl = self::defaultLaunchUrl($provider, $address);
            if ($launchUrl !== '') {
                $attributes['remote_access_launch_url'] = $launchUrl;
            }
        }

        if ($webUrl === '') {
            $webUrl = self::defaultWebUrl($provider, $address, $launchUrl);
            if ($webUrl !== '') {
                $attributes['remote_access_web_url'] = $webUrl;
            }
        }

        return $attributes;
    }

    /**
     * @param  mixed  $payload
     * @return array<string, string>|null
     */
    public static function normalizeSnapshot(mixed $payload): ?array
    {
        if (! is_array($payload)) {
            return null;
        }

        $provider = trim((string) ($payload['provider'] ?? ''));
        $address = trim((string) ($payload['address'] ?? ''));
        $username = trim((string) ($payload['username'] ?? ''));
        $port = self::normalizePortValue($payload['port'] ?? null, $provider);
        $password = trim((string) ($payload['password'] ?? ''));
        $launchUrl = trim((string) ($payload['launch_url'] ?? ''));
        $webUrl = trim((string) ($payload['web_url'] ?? ''));
        $notes = trim((string) ($payload['notes'] ?? ''));

        if ($provider === '' || $address === '') {
            return null;
        }

        if ($launchUrl === '') {
            $launchUrl = self::defaultLaunchUrl($provider, $address);
        }

        if ($webUrl === '') {
            $webUrl = self::defaultWebUrl($provider, $address, $launchUrl);
        } else {
            $webUrl = self::normalizeWebUrl($webUrl);
        }

        return array_filter([
            'provider' => $provider,
            'provider_label' => self::providerLabel($provider),
            'address' => $address,
            'username' => $username,
            'port' => $port !== null ? (string) $port : '',
            'password' => $password,
            'launch_url' => $launchUrl,
            'web_url' => $webUrl,
            'notes' => $notes,
        ], static fn (string $value): bool => $value !== '');
    }

    public static function providerLabel(?string $provider): string
    {
        $provider = trim((string) $provider);
        if ($provider === '') {
            return 'Remote access';
        }

        return self::providers()[$provider] ?? ucfirst($provider);
    }

    public static function defaultLaunchUrl(string $provider, string $address): string
    {
        $provider = trim($provider);
        $address = preg_replace('/\s+/', '', trim($address)) ?: '';

        if ($provider === 'anydesk' && $address !== '') {
            return 'anydesk:'.$address;
        }

        return '';
    }

    public static function defaultWebUrl(string $provider, string $address, string $launchUrl = ''): string
    {
        $provider = trim($provider);
        $address = trim($address);
        $launchUrl = trim($launchUrl);

        if ($launchUrl !== '' && self::isHttpUrl($launchUrl)) {
            return $launchUrl;
        }

        if (self::isHttpUrl($address)) {
            return $address;
        }

        if (in_array($provider, ['rdp', 'rustdesk', 'other'], true) && self::looksLikeHost($address)) {
            return 'https://'.ltrim($address, '/');
        }

        if ($provider === 'teamviewer') {
            return 'https://web.teamviewer.com/';
        }

        return '';
    }

    public static function responseMessageFromSnapshot(?array $snapshot, ?string $fallback = null): ?string
    {
        $normalized = self::normalizeSnapshot($snapshot);
        $fallback = trim((string) $fallback);

        if ($normalized === null) {
            return $fallback !== '' ? $fallback : null;
        }

        $lines = [
            self::providerLabel($normalized['provider'] ?? null),
            __('Connection ID / host').': '.($normalized['address'] ?? ''),
        ];

        if (($normalized['username'] ?? '') !== '') {
            $lines[] = __('Username').': '.$normalized['username'];
        }

        if (($normalized['port'] ?? '') !== '') {
            $lines[] = __('Port').': '.$normalized['port'];
        }

        if (($normalized['password'] ?? '') !== '') {
            $lines[] = __('Password / code').': '.$normalized['password'];
        }

        if (($normalized['notes'] ?? '') !== '') {
            $lines[] = __('Comment').': '.$normalized['notes'];
        }

        if (($normalized['web_url'] ?? '') !== '') {
            $lines[] = __('Browser URL').': '.$normalized['web_url'];
        }

        if ($fallback !== '') {
            $lines[] = $fallback;
        }

        return implode("\n", array_filter($lines));
    }

    /**
     * @return array<string, string>|null
     */
    public static function snapshotFromRequest(RemoteAccessRequest $request): ?array
    {
        return self::normalizeSnapshot($request->connection_payload);
    }

    private static function normalizeWebUrl(string $url): string
    {
        $url = trim($url);

        if ($url === '') {
            return '';
        }

        if (self::isHttpUrl($url)) {
            return $url;
        }

        if (self::looksLikeHost($url)) {
            return 'https://'.ltrim($url, '/');
        }

        return '';
    }

    private static function isHttpUrl(string $value): bool
    {
        return str_starts_with($value, 'http://') || str_starts_with($value, 'https://');
    }

    private static function looksLikeHost(string $value): bool
    {
        $value = trim($value);

        if ($value === '' || str_contains($value, ' ')) {
            return false;
        }

        return (bool) preg_match('/^[a-z0-9][a-z0-9.\\-]*(?::\\d+)?(?:\\/.*)?$/i', $value);
    }

    private static function normalizePortValue(mixed $value, string $provider): ?int
    {
        if ($value === null || $value === '') {
            return $provider === 'rdp' ? 3389 : null;
        }

        $port = (int) $value;

        return $port >= 1 && $port <= 65535 ? $port : null;
    }
}
