<?php

namespace App\Models;

use App\Support\DiskFileIconResolver;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Disk extends Model
{
    use HasFactory;

    /**
     * @var list<string>
     */
    private const IMAGE_EXTENSIONS = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'heic'];

    /**
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'original_name',
        'storage_path',
        'folder',
        'mime_type',
        'extension',
        'size',
        'description',
        'is_public',
        'owner_id',
    ];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'size' => 'integer',
            'is_public' => 'boolean',
            'owner_id' => 'integer',
        ];
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    /**
     * @return array{icon:string,label:string,tone:string,badge_class:string,icon_wrapper_class:string}
     */
    public function iconMeta(): array
    {
        return DiskFileIconResolver::resolve(
            (string) ($this->extension ?? ''),
            (string) ($this->mime_type ?? '')
        );
    }

    public function isImage(): bool
    {
        $extension = strtolower(trim((string) ($this->extension ?? '')));
        $mimeType = strtolower(trim((string) ($this->mime_type ?? '')));

        return in_array($extension, self::IMAGE_EXTENSIONS, true) || str_starts_with($mimeType, 'image/');
    }
}
