<?php

namespace App\Http\Controllers;

use App\Models\RemoteAccessRequest;
use App\Models\User;
use App\Support\ProfileAvatarManager;
use App\Support\RemoteAccessManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Illuminate\View\View;

class UserController extends Controller
{
    public function show(Request $request, User $member): View
    {
        $viewer = $request->user();

        $member->load([
            'manager:id,name',
            'organizationCompany:id,name',
        ]);

        $member->loadCount([
            'deals',
            'assignedTasks',
            'ownedProjects',
        ]);

        $recentDeals = $member->deals()
            ->with('stage')
            ->latest('updated_at')
            ->limit(8)
            ->get(['id', 'title', 'stage_id', 'amount', 'currency', 'updated_at']);

        $recentTasks = $member->assignedTasks()
            ->latest('updated_at')
            ->limit(8)
            ->get(['id', 'title', 'status', 'priority', 'due_at', 'updated_at']);

        $recentProjects = $member->ownedProjects()
            ->latest('updated_at')
            ->limit(8)
            ->get(['id', 'name', 'status', 'progress', 'due_at', 'updated_at']);

        $isOnline = $member->last_seen_at !== null
            && $member->last_seen_at->gte(now()->subMinutes(5));

        $remoteAccessAvailable = Schema::hasTable('remote_access_requests');
        $canRequestRemoteAccess = $remoteAccessAvailable && (int) $viewer->id !== (int) $member->id;
        $latestRemoteAccessRequest = $canRequestRemoteAccess
            ? RemoteAccessRequest::query()
                ->where('requester_id', $viewer->id)
                ->where('recipient_id', $member->id)
                ->latest('id')
                ->first()
            : null;
        $latestRemoteAccessSnapshot = $latestRemoteAccessRequest && $latestRemoteAccessRequest->status === 'approved'
            ? RemoteAccessManager::snapshotFromRequest($latestRemoteAccessRequest)
            : null;

        $data = [
            'member' => $member,
            'recentDeals' => $recentDeals,
            'recentTasks' => $recentTasks,
            'recentProjects' => $recentProjects,
            'isOnline' => $isOnline,
            'roleLabel' => $this->roleLabel((string) $member->role),
            'photoUrl' => ProfileAvatarManager::url($member->profile_photo_path),
            'photoStyle' => ProfileAvatarManager::style(
                $member->profile_photo_focus_x,
                $member->profile_photo_focus_y,
                $member->profile_photo_zoom
            ),
            'initials' => $this->initials((string) $member->name),
            'remoteAccessAvailable' => $remoteAccessAvailable,
            'canRequestRemoteAccess' => $canRequestRemoteAccess,
            'latestRemoteAccessRequest' => $latestRemoteAccessRequest,
            'latestRemoteAccessSnapshot' => $latestRemoteAccessSnapshot,
        ];

        if ($request->boolean('sidepanel') || $request->header('X-Sidepanel') === '1') {
            return view('sidepanel.users.show', $data);
        }

        return view('users.show', $data);
    }

    private function roleLabel(string $role): string
    {
        return match ($role) {
            'admin' => 'Administrator',
            'moderator', 'manager' => 'Moderator',
            'user', 'sales' => 'User',
            default => $role,
        };
    }

    private function initials(string $name): string
    {
        $parts = preg_split('/\s+/u', trim($name)) ?: [];
        $first = $parts[0] ?? '';
        $last = $parts[1] ?? '';
        $initials = mb_strtoupper(mb_substr($first, 0, 1).mb_substr($last, 0, 1));

        if ($initials === '') {
            $initials = mb_strtoupper(mb_substr($name, 0, 1));
        }

        return $initials;
    }
}
