<?php

namespace App\Http\Controllers;

use App\Models\RemoteAccessRequest;
use App\Models\User;
use App\Support\RemoteBrowserGatewayManager;
use App\Support\RemoteAccessManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Illuminate\View\View;

class RemoteAccessRequestController extends Controller
{
    public function show(Request $request, RemoteAccessRequest $remoteAccessRequest, RemoteBrowserGatewayManager $remoteBrowserGatewayManager): View
    {
        abort_unless(Schema::hasTable('remote_access_requests'), 404);

        $actor = $request->user();
        $remoteAccessRequest->loadMissing([
            'requester:id,name,email,job_title',
            'recipient:id,name,email,job_title',
        ]);

        $isParticipant = (int) $remoteAccessRequest->requester_id === (int) $actor->id
            || (int) $remoteAccessRequest->recipient_id === (int) $actor->id;
        abort_unless($isParticipant, 403);

        $connectionSnapshot = RemoteAccessManager::snapshotFromRequest($remoteAccessRequest);
        $counterpart = (int) $remoteAccessRequest->requester_id === (int) $actor->id
            ? $remoteAccessRequest->recipient
            : $remoteAccessRequest->requester;
        $browserSessionUrl = $connectionSnapshot
            ? $remoteBrowserGatewayManager->urlForRequest($actor, $remoteAccessRequest, $connectionSnapshot)
            : null;

        $data = [
            'remoteAccessRequest' => $remoteAccessRequest,
            'connectionSnapshot' => $connectionSnapshot,
            'counterpart' => $counterpart,
            'browserSessionUrl' => $browserSessionUrl,
        ];

        if ($request->boolean('sidepanel') || $request->header('X-Sidepanel') === '1') {
            return view('sidepanel.remote-access.show', $data);
        }

        return view('remote-access.show', $data);
    }

    public function store(Request $request, User $member): RedirectResponse
    {
        if (! Schema::hasTable('remote_access_requests')) {
            return $this->backToOrigin($request)
                ->with('status', __('Remote access requests are not available yet.'));
        }

        $actor = $request->user();

        abort_if((int) $actor->id === (int) $member->id, 422);

        $existingPending = RemoteAccessRequest::query()
            ->where('requester_id', $actor->id)
            ->where('recipient_id', $member->id)
            ->where('status', 'requested')
            ->latest('id')
            ->first();

        if ($existingPending) {
            return $this->backToOrigin($request)
                ->with('status', __('Remote access request already sent.'));
        }

        RemoteAccessRequest::query()->create([
            'requester_id' => $actor->id,
            'recipient_id' => $member->id,
            'status' => 'requested',
            'request_message' => trim((string) $request->input('request_message', '')) ?: null,
            'requested_at' => now(),
        ]);

        return $this->backToOrigin($request)
            ->with('status', __('Remote access request sent.'));
    }

    public function update(Request $request, RemoteAccessRequest $remoteAccessRequest): RedirectResponse
    {
        if (! Schema::hasTable('remote_access_requests')) {
            return redirect()
                ->route('profile.edit', ['section' => 'profile'])
                ->with('status', __('Remote access requests are not available yet.'));
        }

        $actor = $request->user();
        abort_unless((int) $remoteAccessRequest->recipient_id === (int) $actor->id, 403);

        $validated = $request->validate([
            'status' => ['required', 'in:approved,declined'],
            'response_message' => ['nullable', 'string', 'max:2000'],
        ]);

        if ($remoteAccessRequest->status !== 'requested') {
            return redirect()
                ->route('profile.edit', ['section' => 'profile'])
                ->with('status', __('The request has already been processed.'));
        }

        $connectionSnapshot = null;
        if ((string) $validated['status'] === 'approved') {
            $connectionSnapshot = RemoteAccessManager::snapshotForUser($actor);
            $responseMessage = trim((string) ($validated['response_message'] ?? ''));

            if ($responseMessage === '' && $connectionSnapshot === null) {
                return redirect()
                    ->route('profile.edit', ['section' => 'profile'])
                    ->withErrors([
                        'response_message' => __('Configure your remote access details in profile or enter them manually before approval.'),
                    ], 'default')
                    ->withInput();
            }
        }

        $remoteAccessRequest->forceFill([
            'status' => (string) $validated['status'],
            'response_message' => (string) $validated['status'] === 'approved'
                ? RemoteAccessManager::responseMessageFromSnapshot($connectionSnapshot, (string) ($validated['response_message'] ?? ''))
                : (trim((string) ($validated['response_message'] ?? '')) ?: null),
            'connection_payload' => (string) $validated['status'] === 'approved'
                ? $connectionSnapshot
                : null,
            'resolved_at' => now(),
        ])->save();

        $message = $remoteAccessRequest->status === 'approved'
            ? __('Remote access request approved.')
            : __('Remote access request declined.');

        return redirect()
            ->to(route('profile.edit', ['section' => 'profile']).'#remote-access-requests')
            ->with('status', $message);
    }

    private function backToOrigin(Request $request): RedirectResponse
    {
        $redirectTo = trim((string) $request->input('redirect_to', ''));
        if ($redirectTo !== '') {
            return redirect()->to($redirectTo);
        }

        return redirect()->back();
    }
}
