<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileUpdateRequest;
use App\Models\AccessGroup;
use App\Models\CrmModule;
use App\Models\OrganizationCompany;
use App\Models\OrganizationSetting;
use App\Models\RemoteAccessRequest;
use App\Models\Theme;
use App\Models\User;
use App\Support\AccessControl;
use App\Support\ApiTokenPermissionMatrix;
use App\Support\BrandingManager;
use App\Support\CrmModuleManager;
use App\Support\ProfileAvatarManager;
use App\Support\RemoteAccessManager;
use App\Support\TwoFactorAuthenticator;
use App\Support\UpdateCenterManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request, TwoFactorAuthenticator $twoFactorAuthenticator, UpdateCenterManager $updateCenterManager): View
    {
        $user = $request->user();
        $pending = $request->session()->get('two_factor.pending');
        $organizationSetting = OrganizationSetting::current();
        $organizationCompanies = OrganizationCompany::query()
            ->with('parent:id,name')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get([
                'id',
                'name',
                'logo_path',
                'requisites',
                'address',
                'phones',
                'sort_order',
                'parent_id',
            ]);
        $organizationChildCompanies = $organizationCompanies
            ->whereNotNull('parent_id')
            ->values();

        $twoFactorSetup = null;
        if (is_array($pending) && (int) ($pending['user_id'] ?? 0) === $user->id) {
            $secret = (string) ($pending['secret'] ?? '');
            $recoveryCodes = is_array($pending['recovery_codes'] ?? null)
                ? array_values($pending['recovery_codes'])
                : [];

            if ($secret !== '' && $recoveryCodes !== []) {
                $twoFactorSetup = [
                    'secret' => $secret,
                    'otpauth' => $twoFactorAuthenticator->otpauthUri($user, $secret),
                    'qr_url' => $twoFactorAuthenticator->qrCodeUrl($user, $secret),
                    'recovery_codes' => $recoveryCodes,
                ];
            }
        }

        $apiTokens = $user->tokens()
            ->latest('id')
            ->get()
            ->map(function ($token): array {
                $abilities = is_array($token->abilities) ? $token->abilities : [];

                return [
                    'id' => $token->id,
                    'name' => $token->name,
                    'abilities' => $abilities,
                    'ability_labels' => ApiTokenPermissionMatrix::labelsForToken($abilities),
                    'last_used_at' => $token->last_used_at,
                    'expires_at' => $token->expires_at,
                    'created_at' => $token->created_at,
                ];
            })
            ->values();

        $managers = User::query()
            ->whereKeyNot($user->id)
            ->orderBy('name')
            ->get(['id', 'name', 'job_title']);

        $canManageAccess = AccessControl::canManageAccess($user);
        $remoteAccessAvailable = Schema::hasTable('remote_access_requests');
        $incomingRemoteAccessRequests = $remoteAccessAvailable
            ? RemoteAccessRequest::query()
                ->with(['requester:id,name,email,job_title,phone'])
                ->where('recipient_id', $user->id)
                ->latest('requested_at')
                ->latest('id')
                ->limit(10)
                ->get()
            : collect();
        $outgoingRemoteAccessRequests = $remoteAccessAvailable
            ? RemoteAccessRequest::query()
                ->with(['recipient:id,name,email,job_title,phone'])
                ->where('requester_id', $user->id)
                ->latest('requested_at')
                ->latest('id')
                ->limit(10)
                ->get()
            : collect();
        $accessUsers = collect();
        $accessGroups = collect();
        $modules = collect();
        $themes = collect();
        $updateSettings = null;
        $updateChecks = collect();
        $updateInstallations = collect();
        $updateStorageReady = true;
        if ($canManageAccess) {
            $accessUsers = User::query()
                ->with([
                    'manager:id,name',
                    'organizationCompany:id,name',
                    'accessGroup:id,name',
                ])
                ->orderBy('name')
                ->get([
                    'id',
                    'name',
                    'email',
                    'phone',
                    'job_title',
                    'manager_id',
                    'organization_company_id',
                    'birth_date',
                    'role',
                    'permissions',
                    'access_group_id',
                    'last_seen_at',
                ]);

            $accessGroups = AccessGroup::query()
                ->withCount('users')
                ->orderBy('name')
                ->get([
                    'id',
                    'name',
                    'permissions',
                    'created_at',
                    'updated_at',
                ]);

            $modules = CrmModule::query()
                ->with('author:id,name')
                ->latest('id')
                ->get();

            $themes = Theme::query()
                ->with('creator:id,name')
                ->orderBy('name')
                ->get();

            $updateStorageReady = $updateCenterManager->storageIsReady();
            $updateSettings = $updateCenterManager->settings();
            if ($updateStorageReady && $updateSettings->exists) {
                $updateChecks = $updateSettings->checks()
                    ->limit(20)
                    ->get();
                $updateInstallations = $updateSettings->installations()
                    ->with('actor:id,name,email')
                    ->limit(20)
                    ->get();
            }
        }

        return view('profile.edit', [
            'user' => $user,
            'managers' => $managers,
            'canManageAccess' => $canManageAccess,
            'accessUsers' => $accessUsers,
            'accessGroups' => $accessGroups,
            'organizationSetting' => $organizationSetting,
            'organizationCompanies' => $organizationCompanies,
            'organizationChildCompanies' => $organizationChildCompanies,
            'accessRoles' => AccessControl::roles(),
            'accessEntities' => AccessControl::entities(),
            'accessActions' => AccessControl::actions(),
            'modules' => $modules,
            'themes' => $themes,
            'updateSettings' => $updateSettings,
            'updateChecks' => $updateChecks,
            'updateInstallations' => $updateInstallations,
            'updateStorageReady' => $updateStorageReady,
            'moduleSupportedHooks' => CrmModuleManager::supportedHooks(),
            'twoFactorSetup' => $twoFactorSetup,
            'apiTokens' => $apiTokens,
            'apiTokenModules' => ApiTokenPermissionMatrix::modules(),
            'apiTokenActions' => ApiTokenPermissionMatrix::actions(),
            'remoteAccessProviders' => RemoteAccessManager::providers(),
            'remoteAccessProfileConfigured' => RemoteAccessManager::hasProfileConnection($user),
            'remoteAccessAvailable' => $remoteAccessAvailable,
            'incomingRemoteAccessRequests' => $incomingRemoteAccessRequests,
            'outgoingRemoteAccessRequests' => $outgoingRemoteAccessRequests,
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request, BrandingManager $brandingManager): RedirectResponse
    {
        $validated = RemoteAccessManager::applyProfileDefaults($request->validated());
        $user = $request->user();
        $previousZoom = ProfileAvatarManager::normalizeZoom($user->profile_photo_zoom);

        $user->fill(Arr::except($validated, [
            'profile_photo',
            'remove_profile_photo',
            'profile_photo_zoom',
        ]));

        if ($user->isDirty('email')) {
            $user->email_verified_at = null;
        }

        $removePhoto = $request->boolean('remove_profile_photo');
        if ($removePhoto && $user->profile_photo_path) {
            Storage::disk('public')->delete($user->profile_photo_path);
            $user->profile_photo_path = null;
        }

        if ($request->hasFile('profile_photo')) {
            $newPath = $request->file('profile_photo')->store('profile-photos', 'public');
            if ($user->profile_photo_path && $user->profile_photo_path !== $newPath) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }

            $user->profile_photo_path = $newPath;
        }

        if ($user->profile_photo_path) {
            $user->profile_photo_focus_x = ProfileAvatarManager::DEFAULT_FOCUS_X;
            $user->profile_photo_focus_y = ProfileAvatarManager::DEFAULT_FOCUS_Y;
            $user->profile_photo_zoom = ProfileAvatarManager::normalizeZoom(
                $validated['profile_photo_zoom'] ?? $previousZoom
            );
        } else {
            $user->profile_photo_focus_x = ProfileAvatarManager::DEFAULT_FOCUS_X;
            $user->profile_photo_focus_y = ProfileAvatarManager::DEFAULT_FOCUS_Y;
            $user->profile_photo_zoom = ProfileAvatarManager::DEFAULT_ZOOM;
        }

        $shouldRefreshBrand = $user->isDirty('organization_company_id');
        $user->save();

        if ($shouldRefreshBrand) {
            $brandingManager->flush();
        }

        $locale = (string) ($user->locale ?? config('app.locale', 'en'));
        App::setLocale($locale);
        $request->session()->put('locale', $locale);

        $redirectSection = (string) $request->query('section', (string) $request->input('_settings_section', ''));
        $redirectParameters = in_array($redirectSection, ['profile', 'localization', 'appearance', 'api', 'updates', 'company', 'users', 'access', 'modules', 'themes'], true)
            ? ['section' => $redirectSection]
            : [];

        return Redirect::route('profile.edit', $redirectParameters)->with('status', 'profile-updated');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }
}
