<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Product;
use App\Models\User;
use App\Support\CrmModuleManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class ProductController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Product::class, 'product');
    }

    public function index(Request $request): View
    {
        $search = trim((string) $request->input('q', ''));
        $status = trim((string) $request->input('status', ''));

        $products = Product::query()
            ->with(['company', 'owner'])
            ->when($search !== '', function ($query) use ($search): void {
                $query->where(function ($sub) use ($search): void {
                    $sub->where('name', 'like', "%{$search}%")
                        ->orWhere('sku', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            })
            ->when(in_array($status, ['active', 'archived'], true), fn ($query) => $query->where('status', $status))
            ->latest('id')
            ->paginate(20)
            ->withQueryString();

        return view('products.index', [
            'products' => $products,
            'search' => $search,
            'status' => $status,
            'statusOptions' => $this->statusOptions(),
        ]);
    }

    public function create(): View
    {
        return view('products.create', [
            'owners' => User::query()->orderBy('name')->get(),
            'companies' => Company::query()->orderBy('name')->get(),
            'statusOptions' => $this->statusOptions(),
        ]);
    }

    public function store(Request $request, CrmModuleManager $moduleManager): RedirectResponse
    {
        $payload = $this->validatedData($request);
        $payload = $moduleManager->applyPayloadHooks('products.store', $payload, [
            'hook' => 'products.store',
            'user_id' => $request->user()->id,
        ], array_keys($payload));

        $product = Product::create($payload);

        return redirect()
            ->route('products.show', $product)
            ->with('success', __('Product has been created.'));
    }

    public function show(Product $product): View
    {
        $product->load(['company', 'owner']);

        return view('products.show', compact('product'));
    }

    public function edit(Product $product): View
    {
        return view('products.edit', [
            'product' => $product,
            'owners' => User::query()->orderBy('name')->get(),
            'companies' => Company::query()->orderBy('name')->get(),
            'statusOptions' => $this->statusOptions(),
        ]);
    }

    public function update(Request $request, Product $product, CrmModuleManager $moduleManager): RedirectResponse
    {
        $payload = $this->validatedData($request, $product);
        $payload = $moduleManager->applyPayloadHooks('products.update', $payload, [
            'hook' => 'products.update',
            'user_id' => $request->user()->id,
            'product_id' => $product->id,
        ], array_keys($payload));

        $product->update($payload);

        return redirect()
            ->route('products.show', $product)
            ->with('success', __('Product has been updated.'));
    }

    public function destroy(Product $product): RedirectResponse
    {
        $product->delete();

        return redirect()
            ->route('products.index')
            ->with('success', __('Product has been deleted.'));
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedData(Request $request, ?Product $product = null): array
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'sku' => [
                'nullable',
                'string',
                'max:100',
                Rule::unique('products', 'sku')->ignore($product?->id),
            ],
            'price' => ['nullable', 'numeric', 'min:0'],
            'currency' => ['required', 'string', 'size:3'],
            'unit' => ['required', 'string', 'max:32'],
            'stock' => ['nullable', 'numeric', 'min:0'],
            'status' => ['required', Rule::in(['active', 'archived'])],
            'company_id' => ['nullable', 'exists:companies,id'],
            'owner_id' => ['nullable', 'exists:users,id'],
            'description' => ['nullable', 'string'],
        ]);

        $validated['price'] = (float) ($validated['price'] ?? 0);
        $validated['stock'] = (float) ($validated['stock'] ?? 0);
        $validated['currency'] = strtoupper((string) $validated['currency']);

        return $validated;
    }

    /**
     * @return array<string, string>
     */
    private function statusOptions(): array
    {
        return [
            'active' => __('Active'),
            'archived' => __('Archived'),
        ];
    }
}
