<?php

namespace App\Http\Controllers;

use App\Models\Pipeline;
use App\Models\User;
use App\Support\CrmModuleManager;
use App\Support\SectionAccessManager;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class PipelineController extends Controller
{
    /**
     * @var list<string>
     */
    private const STAGE_COLOR_PALETTE = [
        '#3B82F6',
        '#0EA5E9',
        '#8B5CF6',
        '#F59E0B',
        '#14B8A6',
        '#6366F1',
        '#EC4899',
        '#94A3B8',
    ];

    public function __construct()
    {
        $this->authorizeResource(Pipeline::class, 'pipeline');
    }

    public function index(Request $request, SectionAccessManager $sectionAccessManager): View
    {
        /** @var User $user */
        $user = $request->user();

        $pipelines = Pipeline::query()
            ->with(['stages', 'deals'])
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $canManageSectionAccess = $sectionAccessManager->canManage($user, 'pipelines');
        $sectionAccessUsers = $canManageSectionAccess
            ? User::query()->orderBy('name')->get(['id', 'name', 'email', 'role', 'permissions'])
            : collect();

        return view('pipelines.index', compact('pipelines', 'canManageSectionAccess', 'sectionAccessUsers'));
    }

    public function create(): View
    {
        return view('pipelines.create');
    }

    public function store(Request $request, CrmModuleManager $moduleManager): RedirectResponse
    {
        $validated = $this->validatedData($request);
        $validated = $moduleManager->applyPayloadHooks('pipelines.store', $validated, [
            'hook' => 'pipelines.store',
            'user_id' => $request->user()->id,
        ], array_keys($validated));

        if ($validated['is_default']) {
            Pipeline::query()->update(['is_default' => false]);
        }

        $pipeline = Pipeline::create([
            'name' => $validated['name'],
            'description' => $validated['description'],
            'creator_id' => $request->user()?->id,
            'is_default' => $validated['is_default'],
            'is_active' => $validated['is_active'],
            'sort_order' => $validated['sort_order'],
        ]);

        $this->syncStages($pipeline, $validated['stages'], $validated['stage_colors']);

        if ($request->boolean('redirect_to_deals')) {
            return redirect()
                ->route('deals.index', ['pipeline_id' => $pipeline->id])
                ->with('success', 'The funnel has been created.');
        }

        return redirect()
            ->route('pipelines.edit', $pipeline)
            ->with('success', 'The funnel has been created.');
    }

    public function show(Pipeline $pipeline): RedirectResponse
    {
        return redirect()->route('pipelines.edit', $pipeline);
    }

    public function edit(Pipeline $pipeline): View
    {
        $pipeline->load(['stages' => fn ($query) => $query->orderBy('sort_order')]);

        $stages = $pipeline->stages->pluck('name')->implode(PHP_EOL);

        return view('pipelines.edit', compact('pipeline', 'stages'));
    }

    public function update(Request $request, Pipeline $pipeline, CrmModuleManager $moduleManager): RedirectResponse
    {
        $validated = $this->validatedData($request, $pipeline);
        $validated = $moduleManager->applyPayloadHooks('pipelines.update', $validated, [
            'hook' => 'pipelines.update',
            'user_id' => $request->user()->id,
            'pipeline_id' => $pipeline->id,
        ], array_keys($validated));

        if ($validated['is_default']) {
            Pipeline::query()->where('id', '!=', $pipeline->id)->update(['is_default' => false]);
        }

        $pipeline->update([
            'name' => $validated['name'],
            'description' => $validated['description'],
            'is_default' => $validated['is_default'],
            'is_active' => $validated['is_active'],
            'sort_order' => $validated['sort_order'],
        ]);

        $this->syncStages($pipeline, $validated['stages'], $validated['stage_colors']);

        if ($request->boolean('redirect_to_deals')) {
            return redirect()
                ->route('deals.index', ['pipeline_id' => $pipeline->id])
                ->with('success', 'The funnel has been updated.');
        }

        return redirect()
            ->route('pipelines.edit', $pipeline)
            ->with('success', 'The funnel has been updated.');
    }

    public function updateStages(Request $request, Pipeline $pipeline): RedirectResponse|JsonResponse
    {
        $this->authorize('update', $pipeline);

        if (is_array($request->input('stages'))) {
            $validated = $request->validate([
                'stages' => ['required', 'array', 'min:1'],
                'stages.*.id' => ['required', 'integer'],
                'stages.*.color' => ['nullable', 'string', 'regex:/^#[0-9A-Fa-f]{6}$/'],
            ]);

            $stages = $pipeline->stages()->orderBy('sort_order')->get()->keyBy('id');
            $sortOrder = 0;

            foreach ($validated['stages'] as $stagePayload) {
                $stageId = (int) ($stagePayload['id'] ?? 0);
                $stage = $stages->get($stageId);
                if (! $stage) {
                    continue;
                }

                $stage->update([
                    'sort_order' => $sortOrder,
                    'color' => $this->resolveStageColor(
                        $stagePayload['color'] ?? null,
                        $stage->color,
                        (bool) $stage->is_won,
                        (bool) $stage->is_lost,
                        $sortOrder
                    ),
                ]);

                $sortOrder++;
                $stages->forget($stageId);
            }

            foreach ($stages as $stage) {
                $stage->update([
                    'sort_order' => $sortOrder,
                ]);
                $sortOrder++;
            }

            $updatedStages = $pipeline->stages()
                ->orderBy('sort_order')
                ->get()
                ->map(fn ($stage): array => [
                    'id' => $stage->id,
                    'name' => $stage->name,
                    'color' => $stage->color,
                    'sort_order' => $stage->sort_order,
                ])
                ->values()
                ->all();

            return response()->json([
                'message' => 'Funnel stages updated.',
                'stages' => $updatedStages,
            ]);
        }

        $validated = $request->validate([
            'stages' => ['nullable', 'string'],
            'stage_colors' => ['nullable', 'array'],
            'stage_colors.*' => ['nullable', 'string', 'regex:/^#[0-9A-Fa-f]{6}$/'],
        ]);

        $stages = trim((string) ($validated['stages'] ?? ''));
        $stageColors = [];
        foreach (($validated['stage_colors'] ?? []) as $stageId => $color) {
            $normalized = $this->normalizeStageColor((string) $color);

            if ($normalized === null) {
                continue;
            }

            $stageColors[(int) $stageId] = $normalized;
        }

        $this->syncStages($pipeline, $stages, $stageColors);

        if ($request->boolean('redirect_to_deals')) {
            return redirect()
                ->route('deals.index', ['pipeline_id' => $pipeline->id])
                ->with('success', 'Funnel stages updated.');
        }

        return back()->with('success', 'Funnel stages updated.');
    }

    public function destroy(Pipeline $pipeline): RedirectResponse
    {
        if ($pipeline->deals()->exists()) {
            return redirect()
                ->route('pipelines.index')
                ->with('error', 'You cannot delete a funnel that contains deals.');
        }

        $wasDefault = $pipeline->is_default;

        $pipeline->delete();

        if ($wasDefault) {
            $replacement = Pipeline::query()->oldest('sort_order')->first();

            if ($replacement) {
                $replacement->update(['is_default' => true]);
            }
        }

        return redirect()
            ->route('pipelines.index')
            ->with('success', 'The funnel has been removed.');
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedData(Request $request, ?Pipeline $pipeline = null): array
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('pipelines', 'name')->ignore($pipeline?->id),
            ],
            'description' => ['nullable', 'string'],
            'is_default' => ['nullable', 'boolean'],
            'is_active' => ['nullable', 'boolean'],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'stages' => ['nullable', 'string'],
            'stage_colors' => ['nullable', 'array'],
            'stage_colors.*' => ['nullable', 'string', 'regex:/^#[0-9A-Fa-f]{6}$/'],
        ]);

        $validated['is_default'] = $request->boolean('is_default');
        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['sort_order'] = (int) ($validated['sort_order'] ?? 0);
        $validated['description'] = $validated['description'] ?? null;
        $validated['stages'] = trim((string) ($validated['stages'] ?? ''));
        $validated['stage_colors'] = collect($validated['stage_colors'] ?? [])
            ->mapWithKeys(function ($color, $stageId): array {
                $normalized = $this->normalizeStageColor((string) $color);

                if ($normalized === null) {
                    return [];
                }

                return [(int) $stageId => $normalized];
            })
            ->all();

        return $validated;
    }

    /**
     * @param  array<int, string>  $stageColorsById
     */
    private function syncStages(Pipeline $pipeline, string $rawStages, array $stageColorsById = []): void
    {
        $stageNames = collect(preg_split('/\r\n|\r|\n/', $rawStages ?: ''))
            ->map(fn ($line) => trim((string) $line))
            ->filter()
            ->unique(fn ($line) => Str::lower($line))
            ->values();

        if ($stageNames->isEmpty()) {
            $stageNames = collect([
                'New lead',
                'Qualification',
                'Offer',
                'Negotiation',
                'Success',
                'Failure',
            ]);
        }

        $existing = $pipeline->stages()->get()->keyBy(fn ($stage) => Str::lower($stage->name));
        $sortOrder = 0;
        $wonExists = false;
        $lostExists = false;

        foreach ($stageNames as $name) {
            $normalized = Str::lower($name);
            $isWon = str_contains($normalized, 'won') || str_contains($normalized, 'usp');
            $isLost = str_contains($normalized, 'lost') || str_contains($normalized, 'proig') || str_contains($normalized, 'refusal');

            $wonExists = $wonExists || $isWon;
            $lostExists = $lostExists || $isLost;

            $code = Str::slug($name, '_');
            if ($code === '') {
                $code = 'stage';
            }
            $code .= '_'.($sortOrder + 1);

            $stage = $existing->pull($normalized);

            $requestedColor = $stage ? ($stageColorsById[$stage->id] ?? null) : null;
            $payload = [
                'name' => $name,
                'code' => $code,
                'sort_order' => $sortOrder,
                'probability' => min(100, $sortOrder * 20),
                'color' => $this->resolveStageColor(
                    $requestedColor,
                    $stage?->color,
                    $isWon,
                    $isLost,
                    $sortOrder
                ),
                'is_won' => $isWon,
                'is_lost' => $isLost,
            ];

            if ($stage) {
                $stage->update($payload);
            } else {
                $pipeline->stages()->create($payload);
            }

            $sortOrder++;
        }

        if (! $wonExists) {
            $pipeline->stages()->create([
                'name' => 'Success',
                'code' => 'won_'.($sortOrder + 1),
                'sort_order' => $sortOrder,
                'probability' => 100,
                'color' => '#10B981',
                'is_won' => true,
                'is_lost' => false,
            ]);
            $sortOrder++;
        }

        if (! $lostExists) {
            $pipeline->stages()->create([
                'name' => 'Failure',
                'code' => 'lost_'.($sortOrder + 1),
                'sort_order' => $sortOrder,
                'probability' => 0,
                'color' => '#EF4444',
                'is_won' => false,
                'is_lost' => true,
            ]);
        }

        foreach ($existing as $stage) {
            if (! $stage->deals()->exists()) {
                $stage->delete();
            }
        }
    }

    private function resolveStageColor(
        ?string $requestedColor,
        ?string $existingColor,
        bool $isWon,
        bool $isLost,
        int $sortOrder
    ): string {
        $normalizedRequested = $this->normalizeStageColor($requestedColor);
        if ($normalizedRequested !== null) {
            return $normalizedRequested;
        }

        $normalizedExisting = $this->normalizeStageColor($existingColor);
        if ($normalizedExisting !== null) {
            return $normalizedExisting;
        }

        if ($isWon) {
            return '#10B981';
        }

        if ($isLost) {
            return '#EF4444';
        }

        return self::STAGE_COLOR_PALETTE[$sortOrder % count(self::STAGE_COLOR_PALETTE)];
    }

    private function normalizeStageColor(?string $color): ?string
    {
        if (! is_string($color)) {
            return null;
        }

        $color = trim($color);
        if (! preg_match('/^#[0-9A-Fa-f]{6}$/', $color)) {
            return null;
        }

        return strtoupper($color);
    }
}
