<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Support\TwoFactorAuthenticator;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class TwoFactorChallengeController extends Controller
{
    public function create(Request $request): View|RedirectResponse
    {
        if (! $this->hasPendingLogin($request)) {
            return redirect()->route('login')->with('status', __('Two-factor challenge session expired. Please log in again.'));
        }

        return view('auth.two-factor-challenge');
    }

    public function store(Request $request, TwoFactorAuthenticator $twoFactorAuthenticator): RedirectResponse
    {
        $validated = $request->validate([
            'code' => ['required', 'string', 'max:30'],
        ]);

        $pendingLogin = $request->session()->get('two_factor_login');

        if (! is_array($pendingLogin) || empty($pendingLogin['user_id'])) {
            return redirect()->route('login')->with('status', __('Two-factor challenge session expired. Please log in again.'));
        }

        $user = User::query()->find((int) $pendingLogin['user_id']);
        if (! $user || ! $user->hasTwoFactorEnabled()) {
            $request->session()->forget('two_factor_login');

            return redirect()->route('login')->with('status', __('Two-factor challenge session expired. Please log in again.'));
        }

        $code = (string) $validated['code'];

        if (! $twoFactorAuthenticator->verifyCode((string) $user->two_factor_secret, $code) && ! $this->consumeRecoveryCode($user, $code)) {
            throw ValidationException::withMessages([
                'code' => __('Invalid two-factor code.'),
            ]);
        }

        $remember = (bool) ($pendingLogin['remember'] ?? false);

        Auth::login($user, $remember);
        $request->session()->forget('two_factor_login');
        $request->session()->regenerate();

        return redirect()->intended(route('dashboard', absolute: false));
    }

    private function hasPendingLogin(Request $request): bool
    {
        $pendingLogin = $request->session()->get('two_factor_login');

        return is_array($pendingLogin) && ! empty($pendingLogin['user_id']);
    }

    private function consumeRecoveryCode(User $user, string $code): bool
    {
        $normalized = strtoupper(str_replace([' ', '-'], '', trim($code)));
        if ($normalized === '') {
            return false;
        }

        $recoveryCodes = $user->two_factor_recovery_codes;

        if (! is_array($recoveryCodes) || $recoveryCodes === []) {
            return false;
        }

        foreach ($recoveryCodes as $index => $storedCode) {
            $storedNormalized = strtoupper(str_replace([' ', '-'], '', (string) $storedCode));

            if (! hash_equals($storedNormalized, $normalized)) {
                continue;
            }

            unset($recoveryCodes[$index]);

            $user->forceFill([
                'two_factor_recovery_codes' => array_values($recoveryCodes),
            ])->save();

            return true;
        }

        return false;
    }
}
