<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\CompanyResource;
use App\Models\Company;
use App\Support\CrmModuleManager;
use Illuminate\Http\UploadedFile;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;

class CompanyController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Company::class, 'company');
    }

    public function index(Request $request)
    {
        $search = trim((string) $request->input('q', ''));

        $companies = Company::query()
            ->with('owner')
            ->withCount(['contacts', 'deals'])
            ->when($search !== '', function ($query) use ($search): void {
                $query->where(function ($sub) use ($search): void {
                    $sub->where('name', 'like', "%{$search}%")
                        ->orWhere('industry', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                });
            })
            ->latest()
            ->paginate(20)
            ->withQueryString();

        return CompanyResource::collection($companies);
    }

    public function store(Request $request, CrmModuleManager $moduleManager): CompanyResource
    {
        $validated = $this->validatedData($request);
        $validated = $moduleManager->applyPayloadHooks('companies.store', $validated, [
            'hook' => 'companies.store',
            'user_id' => $request->user()->id,
            'source' => 'api',
        ], array_keys($validated));
        $avatar = $request->file('avatar');
        $company = Company::create(Arr::except($validated, ['avatar', 'remove_avatar']));
        if ($avatar instanceof UploadedFile) {
            $company->avatar_path = $avatar->store('company-avatars', 'public');
            $company->save();
        }

        return CompanyResource::make($company->load('owner'));
    }

    public function show(Company $company): CompanyResource
    {
        $company->load(['owner']);
        $company->loadCount(['contacts', 'deals']);

        return CompanyResource::make($company);
    }

    public function update(Request $request, Company $company, CrmModuleManager $moduleManager): CompanyResource
    {
        $validated = $this->validatedData($request);
        $validated = $moduleManager->applyPayloadHooks('companies.update', $validated, [
            'hook' => 'companies.update',
            'user_id' => $request->user()->id,
            'company_id' => $company->id,
            'source' => 'api',
        ], array_keys($validated));
        $avatar = $request->file('avatar');
        $removeAvatar = $request->boolean('remove_avatar');

        $company->update(Arr::except($validated, ['avatar', 'remove_avatar']));
        $this->syncAvatar($company, $avatar, $removeAvatar);

        return CompanyResource::make($company->load('owner'));
    }

    public function destroy(Company $company)
    {
        $this->deleteAvatarIfExists($company->avatar_path);
        $company->delete();

        return response()->noContent();
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedData(Request $request): array
    {
        return $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'industry' => ['nullable', 'string', 'max:255'],
            'website' => ['nullable', 'url', 'max:255'],
            'phone' => ['nullable', 'string', 'max:50'],
            'email' => ['nullable', 'email', 'max:255'],
            'address' => ['nullable', 'string'],
            'owner_id' => ['nullable', 'exists:users,id'],
            'source' => ['nullable', 'string', 'max:255'],
            'status' => ['required', 'in:lead,client,partner,inactive'],
            'notes' => ['nullable', 'string'],
            'avatar' => ['nullable', 'image', 'max:10240'],
            'remove_avatar' => ['nullable', 'boolean'],
        ]);
    }

    private function syncAvatar(Company $company, ?UploadedFile $avatar, bool $removeAvatar): void
    {
        $oldPath = $company->avatar_path;

        if ($removeAvatar) {
            $company->avatar_path = null;
        }

        if ($avatar instanceof UploadedFile) {
            $company->avatar_path = $avatar->store('company-avatars', 'public');
        }

        $company->save();

        if (
            is_string($oldPath)
            && $oldPath !== ''
            && $oldPath !== $company->avatar_path
        ) {
            $this->deleteAvatarIfExists($oldPath);
        }
    }

    private function deleteAvatarIfExists(?string $path): void
    {
        if (! is_string($path) || trim($path) === '') {
            return;
        }

        Storage::disk('public')->delete($path);
    }
}
