@php
    $task = $task ?? null;
    $taskStatusOptions = $taskStatusOptions ?? [
        'todo' => __('To do'),
        'in_progress' => __('In progress'),
        'review' => __('Review'),
        'done' => __('Done'),
    ];
    $selectedProjectId = (int) old('project_id', request('project_id', $requestedProjectId ?? $task?->project_id));
    $selectedProjectStageId = (int) old('project_stage_id', $requestedProjectStageId ?? $task?->project_stage_id);
    $selectedParentId = (int) old('parent_id', request('parent_id', $requestedParentId ?? $task?->parent_id));

    $projectStageMap = $projects->mapWithKeys(fn ($project) => [
        $project->id => $project->stages->map(fn ($stage) => ['id' => $stage->id, 'name' => $stage->name])->values(),
    ]);
@endphp

<div class="grid grid-cols-1 md:grid-cols-2 gap-4">
    <div class="md:col-span-2">
        <label for="title" class="block text-sm font-medium text-gray-700">Task name</label>
        <input id="title" name="title" type="text" required value="{{ old('title', $task?->title) }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
    </div>

    <div>
        <label for="parent_id" class="block text-sm font-medium text-gray-700">Parent task (subtask)</label>
        <select id="parent_id" name="parent_id" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            <option value="">Without parent</option>
            @foreach ($parentTasks as $parentTask)
                @continue($task && $parentTask->id === $task->id)
                <option value="{{ $parentTask->id }}" @selected($selectedParentId === $parentTask->id)>#{{ $parentTask->id }} · {{ $parentTask->title }}</option>
            @endforeach
        </select>
    </div>

    <div>
        <label for="assignee_id" class="block text-sm font-medium text-gray-700">Executor</label>
        <x-user-search-select
            id="assignee_id"
            name="assignee_id"
            :users="$users"
            :selected="old('assignee_id', $task?->assignee_id)"
            placeholder="Search for an artist"
            empty-label="Not assigned"
        />
    </div>

    <div>
        <label for="status" class="block text-sm font-medium text-gray-700">Status</label>
        <select id="status" name="status" required class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            @foreach ($taskStatusOptions as $value => $label)
                <option value="{{ $value }}" @selected(old('status', $task?->status ?? 'todo') === $value)>{{ $label }}</option>
            @endforeach
        </select>
    </div>

    <div>
        <label for="priority" class="block text-sm font-medium text-gray-700">Priority</label>
        <select id="priority" name="priority" required class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            @foreach (['low' => 'Low', 'medium' => 'Medium', 'high' => 'High', 'urgent' => 'Urgent'] as $value => $label)
                <option value="{{ $value }}" @selected(old('priority', $task?->priority ?? 'medium') === $value)>{{ $label }}</option>
            @endforeach
        </select>
    </div>

    <div>
        <label for="starts_at" class="block text-sm font-medium text-gray-700">Start</label>
        <input id="starts_at" name="starts_at" type="datetime-local" value="{{ old('starts_at', $task?->starts_at?->format('Y-m-d\TH:i')) }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
    </div>

    <div>
        <label for="due_at" class="block text-sm font-medium text-gray-700">Term</label>
        <input id="due_at" name="due_at" type="datetime-local" value="{{ old('due_at', request('due_at', $task?->due_at?->format('Y-m-d\TH:i'))) }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
    </div>

    <div>
        <label for="reminder_at" class="block text-sm font-medium text-gray-700">Reminder</label>
        <input id="reminder_at" name="reminder_at" type="datetime-local" value="{{ old('reminder_at', $task?->reminder_at?->format('Y-m-d\TH:i')) }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
    </div>

    <div>
        <label for="deal_id" class="block text-sm font-medium text-gray-700">Deal</label>
        <select id="deal_id" name="deal_id" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            <option value="">No binding</option>
            @foreach ($deals as $deal)
                <option value="{{ $deal->id }}" @selected((int) old('deal_id', request('deal_id', $task?->deal_id)) === $deal->id)>{{ $deal->title }}</option>
            @endforeach
        </select>
    </div>

    <div>
        <label for="company_id" class="block text-sm font-medium text-gray-700">Company</label>
        <select id="company_id" name="company_id" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            <option value="">No binding</option>
            @foreach ($companies as $company)
                <option value="{{ $company->id }}" @selected((int) old('company_id', request('company_id', $task?->company_id)) === $company->id)>{{ $company->name }}</option>
            @endforeach
        </select>
    </div>

    <div>
        <label for="project_id" class="block text-sm font-medium text-gray-700">Project</label>
        <select id="project_id" name="project_id" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            <option value="">Without a project</option>
            @foreach ($projects as $project)
                <option value="{{ $project->id }}" @selected($selectedProjectId === $project->id)>{{ $project->name }}</option>
            @endforeach
        </select>
    </div>

    <div>
        <label for="project_stage_id" class="block text-sm font-medium text-gray-700">Project stage</label>
        <select id="project_stage_id" name="project_stage_id" data-selected="{{ $selectedProjectStageId }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            <option value="">No stage</option>
            @foreach (($projects->firstWhere('id', $selectedProjectId)?->stages ?? collect()) as $stage)
                <option value="{{ $stage->id }}" @selected($selectedProjectStageId === $stage->id)>{{ $stage->name }}</option>
            @endforeach
        </select>
    </div>

    <div class="md:col-span-2">
        <label for="contact_id" class="block text-sm font-medium text-gray-700">Contact</label>
        <select id="contact_id" name="contact_id" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            <option value="">No binding</option>
            @foreach ($contacts as $contact)
                <option value="{{ $contact->id }}" @selected((int) old('contact_id', request('contact_id', $task?->contact_id)) === $contact->id)>{{ $contact->full_name ?: $contact->first_name }}</option>
            @endforeach
        </select>
    </div>

    <div>
        <label for="estimated_hours" class="block text-sm font-medium text-gray-700">Hours plan</label>
        <input id="estimated_hours" name="estimated_hours" type="number" min="0" step="0.1" value="{{ old('estimated_hours', $task?->estimated_hours ?? 0) }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
    </div>

    <div>
        <label for="tracked_hours" class="block text-sm font-medium text-gray-700">Clock fact</label>
        <input id="tracked_hours" name="tracked_hours" type="number" min="0" step="0.1" value="{{ old('tracked_hours', $task?->tracked_hours ?? 0) }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
    </div>

    <div>
        <label for="sort_order" class="block text-sm font-medium text-gray-700">Sorting</label>
        <input id="sort_order" name="sort_order" type="number" min="0" step="1" value="{{ old('sort_order', $task?->sort_order ?? 0) }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
    </div>
</div>

<div class="mt-4">
    <label for="description" class="block text-sm font-medium text-gray-700">Description</label>
    <textarea id="description" name="description" rows="4" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">{{ old('description', $task?->description) }}</textarea>
</div>

<script>
    document.addEventListener('DOMContentLoaded', () => {
        const stageMap = @json($projectStageMap);
        const projectSelect = document.getElementById('project_id');
        const stageSelect = document.getElementById('project_stage_id');

        if (!projectSelect || !stageSelect) {
            return;
        }

        const renderProjectStages = () => {
            const projectId = projectSelect.value;
            const stages = stageMap[projectId] || [];
            const selected = stageSelect.dataset.selected || stageSelect.value;

            stageSelect.innerHTML = '';

            const emptyOption = document.createElement('option');
            emptyOption.value = '';
            emptyOption.textContent = 'No stage';
            stageSelect.appendChild(emptyOption);

            stages.forEach((stage) => {
                const option = document.createElement('option');
                option.value = stage.id;
                option.textContent = stage.name;
                if (String(stage.id) === String(selected)) {
                    option.selected = true;
                }
                stageSelect.appendChild(option);
            });

            stageSelect.dataset.selected = stageSelect.value;
        };

        projectSelect.addEventListener('change', () => {
            stageSelect.dataset.selected = '';
            renderProjectStages();
        });

        stageSelect.addEventListener('change', () => {
            stageSelect.dataset.selected = stageSelect.value;
        });

        renderProjectStages();
    });
</script>
