@php
    /** @var \Illuminate\Support\Collection<int, \App\Models\CrmModule> $modules */
    $modules = isset($modules) && $modules instanceof \Illuminate\Support\Collection ? $modules : collect();
    $moduleSupportedHooks = isset($moduleSupportedHooks) && is_array($moduleSupportedHooks) ? $moduleSupportedHooks : [];
@endphp

<section class="space-y-6">
    <header>
        <h2 class="text-lg font-medium text-gray-900 inline-flex items-center gap-2">
            <x-menu-icon name="fa-solid fa-puzzle-piece" class="h-5 w-5 text-gray-500" />
            {{ __('Modules') }}
        </h2>

        <p class="mt-1 text-sm text-gray-600">
            {{ __('Create your own modules, upload ZIP archives and extend CRM behavior through hook scripts.') }}
        </p>
    </header>

    <div class="flex flex-wrap items-center gap-3 rounded-lg border border-slate-200 bg-slate-50 p-4">
        <a
            href="{{ route('docs.modules.page') }}"
            class="inline-flex items-center rounded-md border border-sky-200 bg-sky-50 px-3 py-1.5 text-sm font-medium text-sky-700 transition hover:border-sky-300 hover:bg-sky-100"
        >
            {{ __('Open modules documentation') }}
        </a>

        <a
            href="{{ route('docs.modules') }}"
            class="inline-flex items-center rounded-md border border-slate-300 px-3 py-1.5 text-sm font-medium text-slate-700 transition hover:border-slate-400 hover:text-slate-900"
            target="_blank"
            rel="noopener"
        >
            {{ __('Open docs/modules.md') }}
        </a>
    </div>

    <div class="grid grid-cols-1 gap-4 xl:grid-cols-2">
        <form method="POST" action="{{ route('modules.scaffold') }}" class="space-y-4 rounded-lg border border-gray-200 p-4">
            @csrf

            <div>
                <h3 class="text-sm font-semibold text-gray-900">{{ __('Create module scaffold') }}</h3>
                <p class="mt-1 text-xs text-gray-500">{{ __('Generate a ready-to-archive template with module.json and hook examples.') }}</p>
            </div>

            <div class="space-y-3">
                <div>
                    <label for="module-scaffold-name" class="block text-sm font-medium text-gray-700">{{ __('Module name') }}</label>
                    <input
                        id="module-scaffold-name"
                        type="text"
                        name="name"
                        value="{{ old('name', 'Custom Automation Module') }}"
                        class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                        required
                    >
                    <x-input-error class="mt-2" :messages="$errors->get('name')" />
                </div>

                <div>
                    <label for="module-scaffold-slug" class="block text-sm font-medium text-gray-700">{{ __('Slug') }}</label>
                    <input
                        id="module-scaffold-slug"
                        type="text"
                        name="slug"
                        value="{{ old('slug', 'custom-automation-module') }}"
                        class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                        required
                    >
                    <p class="mt-1 text-xs text-gray-500">{{ __('Use lowercase latin letters, digits, dots, underscores and dashes.') }}</p>
                    <x-input-error class="mt-2" :messages="$errors->get('slug')" />
                </div>

                <div>
                    <label for="module-scaffold-version" class="block text-sm font-medium text-gray-700">{{ __('Version') }}</label>
                    <input
                        id="module-scaffold-version"
                        type="text"
                        name="version"
                        value="{{ old('version', '1.0.0') }}"
                        class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                    >
                    <x-input-error class="mt-2" :messages="$errors->get('version')" />
                </div>

                <div>
                    <label for="module-scaffold-description" class="block text-sm font-medium text-gray-700">{{ __('Description') }}</label>
                    <textarea
                        id="module-scaffold-description"
                        name="description"
                        rows="3"
                        class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                    >{{ old('description', 'Module for custom business automation in CRM entities.') }}</textarea>
                    <x-input-error class="mt-2" :messages="$errors->get('description')" />
                </div>
            </div>

            <div>
                <button type="submit" class="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white hover:bg-indigo-500">
                    {{ __('Download scaffold ZIP') }}
                </button>
            </div>
        </form>

        <form method="POST" action="{{ route('modules.store') }}" enctype="multipart/form-data" class="space-y-4 rounded-lg border border-gray-200 p-4">
            @csrf

            <div>
                <h3 class="text-sm font-semibold text-gray-900">{{ __('Upload module archive') }}</h3>
                <p class="mt-1 text-xs text-gray-500">{{ __('Upload ZIP with module.json in archive root. Existing module with same slug will be updated.') }}</p>
            </div>

            <div>
                <label for="module-archive" class="block text-sm font-medium text-gray-700">{{ __('ZIP archive') }}</label>
                <input
                    id="module-archive"
                    type="file"
                    name="archive"
                    accept=".zip,application/zip,application/x-zip-compressed"
                    class="mt-1 block w-full text-sm text-gray-700 file:mr-4 file:rounded-md file:border-0 file:bg-indigo-50 file:px-3 file:py-2 file:text-sm file:font-medium file:text-indigo-700 hover:file:bg-indigo-100"
                    required
                >
                <x-input-error class="mt-2" :messages="$errors->get('archive')" />
            </div>

            @if ($moduleSupportedHooks !== [])
                <div class="rounded-md border border-slate-200 bg-slate-50 p-3">
                    <p class="text-xs font-semibold uppercase tracking-wide text-slate-600">{{ __('Supported hooks') }}</p>
                    <div class="mt-2 flex flex-wrap gap-1">
                        @foreach ($moduleSupportedHooks as $hook)
                            <code class="rounded bg-white px-2 py-1 text-xs text-slate-700">{{ $hook }}</code>
                        @endforeach
                    </div>
                </div>
            @endif

            <div>
                <button type="submit" class="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white hover:bg-indigo-500">
                    {{ __('Install module') }}
                </button>
            </div>
        </form>
    </div>

    <div class="space-y-3">
        <h3 class="text-sm font-semibold text-gray-900">{{ __('Installed modules') }}</h3>

        @if ($modules->isEmpty())
            <p class="text-sm text-gray-500">{{ __('No modules installed yet.') }}</p>
        @else
            <div class="overflow-hidden rounded-lg border border-gray-200">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-3 py-2 text-left text-xs font-semibold uppercase tracking-wide text-gray-500">{{ __('Module') }}</th>
                            <th class="px-3 py-2 text-left text-xs font-semibold uppercase tracking-wide text-gray-500">{{ __('Hooks') }}</th>
                            <th class="px-3 py-2 text-left text-xs font-semibold uppercase tracking-wide text-gray-500">{{ __('Status') }}</th>
                            <th class="px-3 py-2 text-left text-xs font-semibold uppercase tracking-wide text-gray-500">{{ __('Updated') }}</th>
                            <th class="px-3 py-2 text-left text-xs font-semibold uppercase tracking-wide text-gray-500">{{ __('Actions') }}</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100 bg-white">
                        @foreach ($modules as $module)
                            @php
                                $hooks = is_array($module->hooks) ? array_keys($module->hooks) : [];
                            @endphp
                            <tr>
                                <td class="px-3 py-3 align-top">
                                    <p class="text-sm font-medium text-gray-900">{{ $module->name }}</p>
                                    <p class="text-xs text-gray-500">{{ $module->slug }} · v{{ $module->version }}</p>
                                    @if ($module->description)
                                        <p class="mt-1 text-xs text-gray-600">{{ $module->description }}</p>
                                    @endif
                                </td>

                                <td class="px-3 py-3 align-top">
                                    @if ($hooks === [])
                                        <span class="text-xs text-gray-500">{{ __('No hooks') }}</span>
                                    @else
                                        <div class="flex flex-wrap gap-1">
                                            @foreach ($hooks as $hook)
                                                <code class="rounded bg-gray-100 px-2 py-1 text-xs text-gray-700">{{ $hook }}</code>
                                            @endforeach
                                        </div>
                                    @endif
                                </td>

                                <td class="px-3 py-3 align-top">
                                    <span class="inline-flex items-center rounded-md px-2 py-1 text-xs font-medium {{ $module->is_enabled ? 'bg-emerald-50 text-emerald-700' : 'bg-slate-100 text-slate-700' }}">
                                        {{ $module->is_enabled ? __('Enabled') : __('Disabled') }}
                                    </span>
                                </td>

                                <td class="px-3 py-3 align-top text-xs text-gray-700">
                                    {{ $module->updated_at?->format('d.m.Y H:i') }}
                                    @if ($module->author)
                                        <div class="text-gray-500">{{ __('By') }}: {{ $module->author->name }}</div>
                                    @endif
                                </td>

                                <td class="px-3 py-3 align-top">
                                    <div class="flex flex-wrap items-center gap-2">
                                        <form method="POST" action="{{ route('modules.update', $module) }}">
                                            @csrf
                                            @method('PATCH')
                                            <input type="hidden" name="is_enabled" value="{{ $module->is_enabled ? '0' : '1' }}">
                                            <button type="submit" class="rounded-md border border-gray-300 px-3 py-1.5 text-xs font-medium text-gray-700 hover:bg-gray-50">
                                                {{ $module->is_enabled ? __('Disable') : __('Enable') }}
                                            </button>
                                        </form>

                                        <form method="POST" action="{{ route('modules.destroy', $module) }}" onsubmit="return confirm('{{ __('Delete module?') }}')">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="rounded-md border border-red-300 px-3 py-1.5 text-xs font-medium text-red-600 hover:bg-red-50">
                                                {{ __('Delete') }}
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
        @endif
    </div>
</section>
