<?php

namespace App\Http\Controllers;

use App\Models\TelephonyCall;
use App\Models\TelephonySetting;
use App\Models\User;
use App\Support\CrmModuleManager;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;

class TelephonyWebhookController extends Controller
{
    public function handle(Request $request, CrmModuleManager $moduleManager): JsonResponse
    {
        $settings = TelephonySetting::query()->first();
        if (! $settings || ! $settings->is_active) {
            return response()->json(['status' => 'ignored'], 202);
        }

        $secret = (string) ($settings->webhook_secret ?? '');
        if ($secret !== '') {
            $incomingSecret = (string) ($request->header('X-CRM-Webhook-Secret', $request->input('secret', '')));
            if ($incomingSecret === '' || ! hash_equals($secret, $incomingSecret)) {
                return response()->json(['message' => 'Unauthorized'], 401);
            }
        }

        $payload = is_array($request->all()) ? $request->all() : [];
        $normalized = $this->normalizePayload($payload);

        $existing = null;
        $externalId = $normalized['external_id'] ?? null;
        if ($externalId) {
            $existing = TelephonyCall::query()->where('external_id', $externalId)->first();
        }

        if ($existing) {
            $normalized = $moduleManager->applyPayloadHooks('telephony.update', $normalized, [
                'hook' => 'telephony.update',
                'source' => 'webhook',
                'call_id' => $existing->id,
            ], array_keys($normalized));

            $existing->fill($normalized)->save();

            return response()->json(['status' => 'updated', 'id' => $existing->id]);
        }

        $normalized = $moduleManager->applyPayloadHooks('telephony.store', $normalized, [
            'hook' => 'telephony.store',
            'source' => 'webhook',
        ], array_keys($normalized));

        $call = TelephonyCall::query()->create($normalized);

        return response()->json(['status' => 'created', 'id' => $call->id]);
    }

    /**
     * @param  array<string, mixed>  $payload
     * @return array<string, mixed>
     */
    private function normalizePayload(array $payload): array
    {
        $direction = strtolower(trim((string) ($payload['direction'] ?? '')));
        if (in_array($direction, ['incoming', 'in'], true)) {
            $direction = 'inbound';
        }
        if (in_array($direction, ['outgoing', 'out'], true)) {
            $direction = 'outbound';
        }
        if (! in_array($direction, ['inbound', 'outbound'], true)) {
            $direction = 'inbound';
        }

        $status = strtolower(trim((string) ($payload['status'] ?? 'new')));
        if ($status === '') {
            $status = 'new';
        }

        $externalId = $this->nullableText($payload['external_id'] ?? $payload['id'] ?? $payload['call_id'] ?? null);
        $from = $this->nullableText($payload['from_number'] ?? $payload['from'] ?? $payload['caller'] ?? null);
        $to = $this->nullableText($payload['to_number'] ?? $payload['to'] ?? $payload['callee'] ?? null);
        $recordingUrl = $this->nullableText($payload['recording_url'] ?? $payload['recording'] ?? null);

        $duration = (int) ($payload['duration_seconds'] ?? $payload['duration'] ?? 0);
        $startedAt = $payload['started_at'] ?? $payload['start_time'] ?? null;
        $endedAt = $payload['ended_at'] ?? $payload['end_time'] ?? null;

        $userId = $payload['user_id'] ?? $payload['agent_id'] ?? null;
        if (! $userId && isset($payload['user_email'])) {
            $userId = User::query()->where('email', (string) $payload['user_email'])->value('id');
        }

        $meta = Arr::except($payload, ['secret', 'token', 'api_key']);

        return [
            'external_id' => $externalId,
            'direction' => $direction,
            'from_number' => $from,
            'to_number' => $to,
            'status' => $status,
            'duration_seconds' => max(0, $duration),
            'started_at' => $startedAt,
            'ended_at' => $endedAt,
            'recording_url' => $recordingUrl,
            'user_id' => $userId ? (int) $userId : null,
            'meta' => $meta === [] ? null : $meta,
        ];
    }

    private function nullableText(mixed $value): ?string
    {
        $value = trim((string) $value);

        return $value === '' ? null : $value;
    }
}
