<?php

namespace App\Support;

use App\Models\User;
use App\Models\UserMenuItem;
use Illuminate\Contracts\Cache\Repository as CacheRepository;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class MenuManager
{
    private const ICON_OPTIONS_CACHE_KEY = 'menu-manager.fontawesome-icon-options.v2';

    private const USER_CACHE_VERSION_KEY_PREFIX = 'menu-manager.user-version.v2.';

    private const USER_ITEMS_CACHE_TTL_SECONDS = 86400;

    /**
     * @var array<string, array{route:string,active:string,label:string,icon:string,ability:?string,entity:?string}>
     */
    private const CORE_ITEMS = [
        'news' => ['route' => 'news.index', 'active' => 'news.*', 'label' => 'News', 'icon' => 'fa-solid fa-newspaper', 'ability' => null, 'entity' => null],
        'tasks' => ['route' => 'tasks.index', 'active' => 'tasks.*', 'label' => 'Tasks', 'icon' => 'check-square', 'ability' => null, 'entity' => 'tasks'],
        'projects' => ['route' => 'projects.index', 'active' => 'projects.*', 'label' => 'Projects', 'icon' => 'folder', 'ability' => null, 'entity' => 'projects'],
        'calendar' => ['route' => 'calendar.index', 'active' => 'calendar.*', 'label' => 'Calendar', 'icon' => 'fa-solid fa-calendar-days', 'ability' => 'viewCalendar', 'entity' => 'calendar'],
        'disks' => ['route' => 'disks.index', 'active' => 'disks.*', 'label' => 'Disk', 'icon' => 'fa-solid fa-hard-drive', 'ability' => 'viewDisk', 'entity' => 'disks'],
        'deals' => ['route' => 'deals.index', 'active' => 'deals.*', 'label' => 'Deals', 'icon' => 'briefcase', 'ability' => null, 'entity' => 'deals'],
        'pipelines' => ['route' => 'pipelines.index', 'active' => 'pipelines.*', 'label' => 'Pipelines', 'icon' => 'funnel', 'ability' => null, 'entity' => 'pipelines'],
        'companies' => ['route' => 'companies.index', 'active' => 'companies.*', 'label' => 'Companies', 'icon' => 'building', 'ability' => null, 'entity' => 'companies'],
        'contacts' => ['route' => 'contacts.index', 'active' => 'contacts.*', 'label' => 'Contacts', 'icon' => 'users', 'ability' => null, 'entity' => 'contacts'],
        'forms' => ['route' => 'forms.index', 'active' => 'forms.*', 'label' => 'Forms', 'icon' => 'fa-solid fa-file-lines', 'ability' => 'viewForms', 'entity' => 'forms'],
        'products' => ['route' => 'products.index', 'active' => 'products.*', 'label' => 'Products', 'icon' => 'fa-solid fa-cart-shopping', 'ability' => null, 'entity' => 'products'],
        'warehouses' => ['route' => 'warehouses.index', 'active' => 'warehouses.*', 'label' => 'Warehouses', 'icon' => 'fa-solid fa-warehouse', 'ability' => null, 'entity' => 'warehouses'],
        'activities' => ['route' => 'activities.index', 'active' => 'activities.*', 'label' => 'Activities', 'icon' => 'calendar', 'ability' => null, 'entity' => 'activities'],
        'reports' => ['route' => 'reports.index', 'active' => 'reports.*', 'label' => 'Reports', 'icon' => 'chart-bar', 'ability' => 'viewReports', 'entity' => 'reports'],
        'messengers' => ['route' => 'messengers.index', 'active' => 'messengers.*', 'label' => 'Messengers', 'icon' => 'fa-solid fa-comment-dots', 'ability' => null, 'entity' => 'messengers'],
        'onec' => ['route' => 'onec.index', 'active' => 'onec.*', 'label' => '1C Integration', 'icon' => 'fa-solid fa-database', 'ability' => null, 'entity' => 'onec'],
        'telephony' => ['route' => 'telephony.index', 'active' => 'telephony.*', 'label' => 'Telephony', 'icon' => 'fa-solid fa-phone', 'ability' => null, 'entity' => 'telephony'],
        'mail' => ['route' => 'mail.index', 'active' => 'mail.*', 'label' => 'Mail Service', 'icon' => 'fa-solid fa-envelope', 'ability' => null, 'entity' => 'mail'],
        'hr' => ['route' => 'hr.index', 'active' => 'hr.*', 'label' => 'HR Service', 'icon' => 'fa-solid fa-user-group', 'ability' => null, 'entity' => 'hr'],
        'academy' => ['route' => 'academy.index', 'active' => 'academy.*', 'label' => 'Academy', 'icon' => 'fa-solid fa-graduation-cap', 'ability' => null, 'entity' => null],
        'dashboard' => ['route' => 'dashboard', 'active' => 'dashboard', 'label' => 'Statistics', 'icon' => 'fa-solid fa-chart-line', 'ability' => null, 'entity' => null],
    ];

    /**
     * @var array<string, array{label:string}>
     */
    private const ICON_OPTIONS = [
        'fa-solid fa-house' => ['label' => 'House'],
        'fa-solid fa-building' => ['label' => 'Building'],
        'fa-solid fa-users' => ['label' => 'Users'],
        'fa-solid fa-briefcase' => ['label' => 'Briefcase'],
        'fa-solid fa-list-check' => ['label' => 'Task list'],
        'fa-solid fa-folder' => ['label' => 'Folder'],
        'fa-solid fa-folder-open' => ['label' => 'Open folder'],
        'fa-solid fa-calendar-days' => ['label' => 'Calendar'],
        'fa-solid fa-filter' => ['label' => 'Filter'],
        'fa-solid fa-chart-simple' => ['label' => 'Schedule'],
        'fa-solid fa-chart-line' => ['label' => 'Line'],
        'fa-solid fa-chart-pie' => ['label' => 'Diagram'],
        'fa-solid fa-link' => ['label' => 'Link'],
        'fa-solid fa-star' => ['label' => 'Star'],
        'fa-solid fa-bolt' => ['label' => 'Lightning'],
        'fa-solid fa-gear' => ['label' => 'Settings'],
        'fa-solid fa-globe' => ['label' => 'Globe'],
        'fa-solid fa-earth-europe' => ['label' => 'Europe'],
        'fa-solid fa-phone' => ['label' => 'Telephone'],
        'fa-solid fa-envelope' => ['label' => 'Mail'],
        'fa-solid fa-comment-dots' => ['label' => 'Messages'],
        'fa-solid fa-book' => ['label' => 'Book'],
        'fa-solid fa-database' => ['label' => 'Database'],
        'fa-solid fa-hard-drive' => ['label' => 'Disk'],
        'fa-solid fa-code' => ['label' => 'Code'],
        'fa-solid fa-plug' => ['label' => 'Integrations'],
        'fa-solid fa-cloud' => ['label' => 'Cloud'],
        'fa-solid fa-bell' => ['label' => 'Notifications'],
        'fa-solid fa-file-lines' => ['label' => 'Document'],
        'fa-solid fa-clipboard' => ['label' => 'Buffer'],
        'fa-solid fa-timeline' => ['label' => 'Timeline'],
        'fa-solid fa-flag' => ['label' => 'Flag'],
        'fa-solid fa-lightbulb' => ['label' => 'Idea'],
        'fa-solid fa-graduation-cap' => ['label' => 'Education'],
        'fa-solid fa-screwdriver-wrench' => ['label' => 'Tools'],
        'fa-solid fa-puzzle-piece' => ['label' => 'Puzzle'],
        'fa-solid fa-shop' => ['label' => 'Shop'],
        'fa-solid fa-cart-shopping' => ['label' => 'Basket'],
        'fa-solid fa-truck' => ['label' => 'Delivery'],
        'fa-solid fa-user-plus' => ['label' => 'Add user'],
        'fa-solid fa-address-book' => ['label' => 'Address Book'],
        'fa-solid fa-diagram-project' => ['label' => 'Project'],
        'fa-solid fa-layer-group' => ['label' => 'Layers'],
        'fa-solid fa-sitemap' => ['label' => 'Structure'],
        'fa-solid fa-circle-info' => ['label' => 'Information'],
        'fa-solid fa-headset' => ['label' => 'Support'],
    ];

    /**
     * @var array<string, string>
     */
    private const LEGACY_ICON_CLASS = [
        'home' => 'fa-solid fa-house',
        'building' => 'fa-solid fa-building',
        'users' => 'fa-solid fa-users',
        'briefcase' => 'fa-solid fa-briefcase',
        'check-square' => 'fa-solid fa-list-check',
        'folder' => 'fa-solid fa-folder',
        'calendar' => 'fa-solid fa-calendar-days',
        'funnel' => 'fa-solid fa-filter',
        'chart-bar' => 'fa-solid fa-chart-simple',
        'link' => 'fa-solid fa-link',
        'star' => 'fa-solid fa-star',
        'bolt' => 'fa-solid fa-bolt',
        'settings' => 'fa-solid fa-gear',
        'globe' => 'fa-solid fa-globe',
    ];

    /**
     * @var array<string, string>
     */
    private const STYLE_CLASS_PREFIX = [
        'solid' => 'fa-solid',
        'regular' => 'fa-regular',
        'brands' => 'fa-brands',
    ];

    /**
     * @var array<string, string>
     */
    private const STYLE_LABELS = [
        'solid' => 'Solid',
        'regular' => 'Regular',
        'brands' => 'Brands',
    ];

    public function ensureCoreItems(User $user): void
    {
        $existing = $user->menuItems()
            ->where('is_custom', false)
            ->get()
            ->keyBy('key');

        $newsKey = array_key_first(self::CORE_ITEMS);
        if ($newsKey !== null && ! $existing->has($newsKey) && $existing->isNotEmpty()) {
            $user->menuItems()->increment('sort_order');
            $existing = $user->menuItems()
                ->where('is_custom', false)
                ->get()
                ->keyBy('key');
        }

        foreach (array_values(array_keys(self::CORE_ITEMS)) as $order => $key) {
            if ($existing->has($key)) {
                continue;
            }

            $definition = self::CORE_ITEMS[$key];

            $user->menuItems()->create([
                'key' => $key,
                'icon' => (string) $definition['icon'],
                'sort_order' => $order,
                'is_visible' => true,
                'is_custom' => false,
            ]);
        }
    }

    /**
     * @return Collection<int, array<string, mixed>>
     */
    public function sidebarItems(User $user): Collection
    {
        return $this->cachedItems($user, 'sidebar', true)
            ->filter(fn (array $item): bool => $this->canSeePresentedItem($user, $item))
            ->map(fn (array $item): array => $this->withActiveState($item))
            ->values();
    }

    /**
     * @return Collection<int, array<string, mixed>>
     */
    public function editorItems(User $user): Collection
    {
        return $this->cachedItems($user, 'editor', false)
            ->filter(fn (array $item): bool => $this->canSeePresentedItem($user, $item))
            ->map(fn (array $item): array => $this->withActiveState($item))
            ->values();
    }

    public function forgetUserCaches(int $userId): void
    {
        $versionKey = $this->userVersionCacheKey($userId);
        $cache = $this->hotCache();
        if ($cache->has($versionKey)) {
            $cache->increment($versionKey);

            return;
        }

        $cache->forever($versionKey, 2);
    }

    /**
     * @return array<string, array{label:string}>
     */
    public static function iconOptions(): array
    {
        /** @var array<string, array{label:string}> $options */
        $options = Cache::rememberForever(self::ICON_OPTIONS_CACHE_KEY, static function (): array {
            $resolved = self::resolveFontAwesomeIconOptions();

            return $resolved !== [] ? $resolved : self::ICON_OPTIONS;
        });

        return $options;
    }

    /**
     * @return list<string>
     */
    public static function iconNames(): array
    {
        return array_keys(self::iconOptions());
    }

    public static function iconFaClass(string $icon): string
    {
        $icon = trim($icon);
        if ($icon === '') {
            return 'fa-solid fa-link';
        }

        if (array_key_exists($icon, self::ICON_OPTIONS)) {
            return $icon;
        }

        if (array_key_exists($icon, self::LEGACY_ICON_CLASS)) {
            return self::LEGACY_ICON_CLASS[$icon];
        }

        return str_contains($icon, 'fa-') ? $icon : 'fa-solid fa-link';
    }

    /**
     * @return array<string, array{label:string}>
     */
    private static function resolveFontAwesomeIconOptions(): array
    {
        $metadataPath = base_path('node_modules/@fortawesome/fontawesome-free/metadata/icon-families.json');
        if (! is_readable($metadataPath)) {
            return [];
        }

        $payload = file_get_contents($metadataPath);
        if ($payload === false) {
            return [];
        }

        /** @var mixed $decoded */
        $decoded = json_decode($payload, true);
        if (! is_array($decoded)) {
            return [];
        }

        $options = [];

        foreach ($decoded as $iconName => $iconMeta) {
            if (! is_string($iconName) || $iconName === '' || ! is_array($iconMeta)) {
                continue;
            }

            $label = trim((string) ($iconMeta['label'] ?? ''));
            if ($label === '') {
                $label = Str::headline(str_replace('-', ' ', $iconName));
            }

            $freeStyles = $iconMeta['familyStylesByLicense']['free'] ?? [];
            if (! is_array($freeStyles)) {
                continue;
            }

            foreach ($freeStyles as $styleMeta) {
                if (! is_array($styleMeta)) {
                    continue;
                }

                $style = strtolower((string) ($styleMeta['style'] ?? ''));
                $classPrefix = self::STYLE_CLASS_PREFIX[$style] ?? null;
                if (! $classPrefix) {
                    continue;
                }

                $styleLabel = self::STYLE_LABELS[$style] ?? ucfirst($style);
                $className = sprintf('%s fa-%s', $classPrefix, $iconName);

                $options[$className] = [
                    'label' => sprintf('%s (%s)', $label, $styleLabel),
                ];
            }
        }

        if ($options === []) {
            return [];
        }

        uasort($options, static fn (array $left, array $right): int => strnatcasecmp($left['label'], $right['label']));

        return $options;
    }

    /**
     * @param  array<string, mixed>  $item
     */
    private function canSeePresentedItem(User $user, array $item): bool
    {
        if ((bool) ($item['is_custom'] ?? false)) {
            return true;
        }

        $key = trim((string) ($item['key'] ?? ''));
        if ($key === '') {
            return false;
        }

        return $this->canSeeCoreKey($user, $key);
    }

    private function canSeeCoreKey(User $user, string $key): bool
    {
        $definition = self::CORE_ITEMS[$key] ?? null;
        if (! $definition) {
            return false;
        }

        $ability = $definition['ability'] ?? null;
        if (is_string($ability) && $ability !== '') {
            return $user->can($ability);
        }

        $entity = $definition['entity'] ?? null;
        if (is_string($entity) && $entity !== '') {
            return AccessControl::allows($user, $entity, 'read');
        }

        return true;
    }

    /**
     * @return Collection<int, array<string, mixed>>
     */
    private function cachedItems(User $user, string $scope, bool $onlyVisible): Collection
    {
        $this->ensureCoreItems($user);

        $locale = app()->getLocale();
        $cacheKey = $this->itemsCacheKey($user, $scope, $locale);

        /** @var array<int, array<string, mixed>> $cached */
        $cached = $this->hotCache()->remember(
            $cacheKey,
            now()->addSeconds(self::USER_ITEMS_CACHE_TTL_SECONDS),
            function () use ($user, $onlyVisible): array {
                return $user->menuItems()
                    ->get()
                    ->filter(fn (UserMenuItem $item) => ! $onlyVisible || $item->is_visible)
                    ->map(fn (UserMenuItem $item): array => $this->presentItem($item, false))
                    ->values()
                    ->all();
            }
        );

        return collect($cached);
    }

    private function itemsCacheKey(User $user, string $scope, string $locale): string
    {
        return sprintf(
            'menu-manager.%s.user.%d.locale.%s.v%d',
            $scope,
            $user->id,
            $locale,
            $this->userCacheVersion($user->id)
        );
    }

    private function userCacheVersion(int $userId): int
    {
        return (int) $this->hotCache()->rememberForever(
            $this->userVersionCacheKey($userId),
            static fn (): int => 1
        );
    }

    private function userVersionCacheKey(int $userId): string
    {
        return self::USER_CACHE_VERSION_KEY_PREFIX.$userId;
    }

    private function hotCache(): CacheRepository
    {
        /** @var CacheRepository $store */
        $store = Cache::store((string) config('cache.hot_store', 'hot'));

        return $store;
    }

    /**
     * @param  array<string, mixed>  $item
     * @return array<string, mixed>
     */
    private function withActiveState(array $item): array
    {
        if ((bool) ($item['is_custom'] ?? false)) {
            $url = trim((string) ($item['url'] ?? ''));
            $item['is_active'] = $url !== ''
                && (request()->fullUrl() === $url || request()->url() === $url);

            return $item;
        }

        $activePattern = trim((string) ($item['_active_pattern'] ?? ''));
        $item['is_active'] = $activePattern !== '' && request()->routeIs($activePattern);
        unset($item['_active_pattern']);

        return $item;
    }

    /**
     * @return array<string, mixed>
     */
    private function presentItem(UserMenuItem $item, bool $includeActive = true): array
    {
        if ($item->is_custom || ! $item->key) {
            return [
                'id' => $item->id,
                'is_custom' => true,
                'key' => null,
                'label' => (string) $item->label,
                'url' => (string) $item->url,
                'icon' => self::iconFaClass((string) $item->icon),
                'open_in_new_tab' => (bool) $item->open_in_new_tab,
                'is_visible' => $item->is_visible,
                'is_active' => $includeActive
                    ? (request()->fullUrl() === $item->url || request()->url() === $item->url)
                    : false,
            ];
        }

        $definition = self::CORE_ITEMS[$item->key] ?? null;
        if (! $definition) {
            return [
                'id' => $item->id,
                'is_custom' => false,
                'key' => $item->key,
                'label' => (string) $item->label,
                'url' => '#',
                'icon' => self::iconFaClass((string) $item->icon),
                'open_in_new_tab' => false,
                'is_visible' => false,
                'is_active' => false,
                '_active_pattern' => null,
            ];
        }

        $icon = $item->key === 'dashboard'
            ? (string) ($definition['icon'] ?? 'fa-solid fa-chart-line')
            : (string) ($item->icon ?: (string) ($definition['icon'] ?? 'link'));

        return [
            'id' => $item->id,
            'is_custom' => false,
            'key' => $item->key,
            'label' => __($definition['label'] ?? ''),
            'url' => route($definition['route'] ?? 'dashboard'),
            'icon' => self::iconFaClass($icon),
            'open_in_new_tab' => false,
            'is_visible' => $item->is_visible,
            'is_active' => $includeActive ? request()->routeIs($definition['active'] ?? '') : false,
            '_active_pattern' => (string) ($definition['active'] ?? ''),
        ];
    }
}
