<?php

namespace App\Support;

use App\Models\Disk;
use App\Models\DiskFolder;
use App\Models\User;
use Illuminate\Http\UploadedFile;
use Illuminate\Http\Response as HttpResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use RuntimeException;
use Symfony\Component\HttpFoundation\StreamedResponse;

class DiskFileManager
{
    public function __construct(
        private readonly DiskPreviewBuilder $previewBuilder
    ) {
    }

    /**
     * @param  array<string, mixed>  $attributes
     */
    public function createFromUpload(UploadedFile $file, User $owner, array $attributes = []): Disk
    {
        $folder = $this->normalizeFolderPath((string) ($attributes['folder'] ?? ''));
        $fileMeta = $this->storeUploadedFile($file, $folder);
        $this->ensureFolderCatalog($folder, $owner);

        return Disk::query()->create([
            'name' => $this->normalizeDisplayName((string) ($attributes['name'] ?? ''), $fileMeta['original_name']),
            'original_name' => $fileMeta['original_name'],
            'storage_path' => $fileMeta['storage_path'],
            'folder' => $folder !== '' ? $folder : null,
            'mime_type' => $fileMeta['mime_type'] !== '' ? $fileMeta['mime_type'] : null,
            'extension' => $fileMeta['extension'] !== '' ? $fileMeta['extension'] : null,
            'size' => $fileMeta['size'],
            'description' => $this->normalizeNullableText($attributes['description'] ?? null),
            'is_public' => (bool) ($attributes['is_public'] ?? false),
            'owner_id' => $owner->id,
        ]);
    }

    /**
     * @param  array<string, mixed>  $attributes
     */
    public function updateFromPayload(Disk $disk, array $attributes, ?UploadedFile $newFile = null): Disk
    {
        $folder = array_key_exists('folder', $attributes)
            ? $this->normalizeFolderPath((string) ($attributes['folder'] ?? ''))
            : (string) ($disk->folder ?? '');

        $payload = [
            'folder' => $folder !== '' ? $folder : null,
        ];

        if (array_key_exists('description', $attributes)) {
            $payload['description'] = $this->normalizeNullableText($attributes['description'] ?? null);
        }

        if (array_key_exists('is_public', $attributes)) {
            $payload['is_public'] = (bool) $attributes['is_public'];
        }

        if (array_key_exists('name', $attributes)) {
            $payload['name'] = $this->normalizeDisplayName((string) ($attributes['name'] ?? ''), $disk->original_name);
        }

        if ($newFile) {
            $this->deleteStoredFile($disk);
            $fileMeta = $this->storeUploadedFile($newFile, $folder);

            $payload['original_name'] = $fileMeta['original_name'];
            $payload['storage_path'] = $fileMeta['storage_path'];
            $payload['mime_type'] = $fileMeta['mime_type'] !== '' ? $fileMeta['mime_type'] : null;
            $payload['extension'] = $fileMeta['extension'] !== '' ? $fileMeta['extension'] : null;
            $payload['size'] = $fileMeta['size'];

            if (! array_key_exists('name', $attributes)) {
                $payload['name'] = $this->normalizeDisplayName('', $fileMeta['original_name']);
            }
        }

        if ($folder !== '') {
            $owner = $disk->owner ?? User::query()->find($disk->owner_id);
            if ($owner instanceof User) {
                $this->ensureFolderCatalog($folder, $owner);
            }
        }

        $disk->fill($payload);
        $disk->save();

        return $disk->refresh()->load('owner');
    }

    public function normalizeFolderPath(string $folder): string
    {
        return $this->normalizeFolder($folder);
    }

    public function ensureFolderCatalog(string $folderPath, User $owner): void
    {
        $folderPath = $this->normalizeFolderPath($folderPath);
        if ($folderPath === '') {
            return;
        }

        DiskFolder::query()->updateOrCreate(
            [
                'owner_id' => $owner->id,
                'path' => $folderPath,
            ],
            [
                'name' => $this->folderNameFromPath($folderPath),
                'parent_path' => $this->parentPathFromFolder($folderPath),
            ]
        );
    }

    public function delete(Disk $disk): void
    {
        $this->deleteStoredFile($disk);
        $disk->delete();
    }

    public function downloadResponse(Disk $disk): StreamedResponse
    {
        $downloadName = $this->buildDownloadName($disk);

        return Storage::disk('public')->download($disk->storage_path, $downloadName);
    }

    public function inlineResponse(Disk $disk): StreamedResponse
    {
        $inlineName = $this->buildDownloadName($disk);
        $mimeType = trim((string) $disk->mime_type) !== ''
            ? (string) $disk->mime_type
            : ($this->guessMimeTypeByExtension((string) $disk->extension) ?? 'application/octet-stream');

        return Storage::disk('public')->response(
            $disk->storage_path,
            $inlineName,
            [
                'Content-Type' => $mimeType,
                'Content-Disposition' => sprintf('inline; filename="%s"', str_replace('"', '', $inlineName)),
                'X-Content-Type-Options' => 'nosniff',
            ]
        );
    }

    public function canPreviewInCrm(Disk $disk): bool
    {
        return $this->previewBuilder->canPreviewInCrm($disk);
    }

    /**
     * @return array{kind:string,title:string,html:\Illuminate\Support\HtmlString|null,message:string|null}
     */
    public function buildPreviewPayload(Disk $disk): array
    {
        return $this->previewBuilder->buildPreviewPayload($disk);
    }

    public function wordPreviewSourceResponse(Disk $disk): HttpResponse|StreamedResponse
    {
        return $this->previewBuilder->buildWordPreviewSourceResponse($disk);
    }

    public function spreadsheetPreviewSourceResponse(Disk $disk): HttpResponse|StreamedResponse
    {
        return $this->previewBuilder->buildSpreadsheetPreviewSourceResponse($disk);
    }

    private function deleteStoredFile(Disk $disk): void
    {
        $path = trim((string) $disk->storage_path);
        if ($path !== '') {
            Storage::disk('public')->delete($path);
        }
    }

    /**
     * @return array{storage_path:string,original_name:string,mime_type:string,extension:string,size:int}
     */
    private function storeUploadedFile(UploadedFile $file, string $folder): array
    {
        $directory = $folder !== '' ? 'disk/'.$folder : 'disk';
        $storagePath = $file->store($directory, 'public');
        $storagePath = is_string($storagePath) ? $storagePath : '';
        if ($storagePath === '') {
            throw new RuntimeException('Unable to store uploaded file.');
        }

        $originalName = trim((string) $file->getClientOriginalName());
        if ($originalName === '') {
            $originalName = trim((string) $file->getFilename());
        }

        $mimeType = trim((string) ($file->getClientMimeType() ?: $file->getMimeType() ?: ''));

        $extension = strtolower(trim((string) $file->getClientOriginalExtension()));
        if ($extension === '') {
            $extension = strtolower((string) pathinfo($originalName, PATHINFO_EXTENSION));
        }

        $size = (int) ($file->getSize() ?? 0);
        if ($size <= 0 && $storagePath !== '' && Storage::disk('public')->exists($storagePath)) {
            $size = (int) Storage::disk('public')->size($storagePath);
        }

        return [
            'storage_path' => $storagePath,
            'original_name' => $originalName,
            'mime_type' => $mimeType,
            'extension' => $extension,
            'size' => max(0, $size),
        ];
    }

    private function normalizeFolder(string $folder): string
    {
        $folder = str_replace('\\', '/', trim($folder));
        $folder = str_replace('..', '', $folder);
        $folder = preg_replace('#/+#', '/', $folder) ?? '';

        return trim($folder, '/');
    }

    private function folderNameFromPath(string $path): string
    {
        $path = $this->normalizeFolderPath($path);
        if ($path === '') {
            return 'Folder';
        }

        $segments = explode('/', $path);
        $last = trim((string) end($segments));

        return $last !== '' ? Str::limit($last, 120, '') : 'Folder';
    }

    private function parentPathFromFolder(string $path): ?string
    {
        $path = $this->normalizeFolderPath($path);
        if (! str_contains($path, '/')) {
            return null;
        }

        $parent = trim((string) dirname($path), '/.');

        return $parent !== '' ? $parent : null;
    }

    private function normalizeDisplayName(string $name, string $fallbackOriginalName): string
    {
        $name = trim($name);

        if ($name === '') {
            $name = trim((string) pathinfo($fallbackOriginalName, PATHINFO_FILENAME));
        }

        if ($name === '') {
            $name = trim($fallbackOriginalName);
        }

        $name = trim(Str::limit($name, 255, ''));

        return $name !== '' ? $name : 'File';
    }

    private function normalizeNullableText(mixed $value): ?string
    {
        $text = trim((string) $value);

        return $text !== '' ? Str::limit($text, 5000, '') : null;
    }

    private function buildDownloadName(Disk $disk): string
    {
        $baseName = trim((string) $disk->name);
        if ($baseName === '') {
            $baseName = trim((string) pathinfo((string) $disk->original_name, PATHINFO_FILENAME));
        }

        if ($baseName === '') {
            $baseName = 'file';
        }

        $extension = strtolower(trim((string) $disk->extension));
        if ($extension !== '' && ! str_ends_with(strtolower($baseName), '.'.$extension)) {
            $baseName .= '.'.$extension;
        }

        return $baseName;
    }

    private function guessMimeTypeByExtension(string $extension): ?string
    {
        return match (strtolower(trim($extension))) {
            'pdf' => 'application/pdf',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'doc' => 'application/msword',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'xls' => 'application/vnd.ms-excel',
            'csv' => 'text/csv',
            default => null,
        };
    }
}
