<?php

namespace App\Policies;

use App\Models\Task;
use App\Models\User;
use App\Support\AccessControl;

class TaskPolicy
{
    public function viewAny(User $user): bool
    {
        return AccessControl::allows($user, 'tasks', 'read');
    }

    public function view(User $user, Task $task): bool
    {
        $decision = AccessControl::overridePermissionState($user, 'tasks', 'read');
        if ($decision !== null) {
            return $decision;
        }

        if (!AccessControl::roleAllows($user, 'tasks', 'read')) {
            return false;
        }

        return AccessControl::isElevated($user)
            || $task->creator_id === $user->id
            || $task->assignee_id === $user->id
            || $this->isProjectMember($user, $task);
    }

    public function create(User $user): bool
    {
        return AccessControl::allows($user, 'tasks', 'create');
    }

    public function update(User $user, Task $task): bool
    {
        $decision = AccessControl::overridePermissionState($user, 'tasks', 'update');
        if ($decision !== null) {
            return $decision;
        }

        if (!AccessControl::roleAllows($user, 'tasks', 'update')) {
            return false;
        }

        return AccessControl::isElevated($user)
            || $task->creator_id === $user->id
            || $task->assignee_id === $user->id
            || $this->isProjectMember($user, $task);
    }

    public function complete(User $user, Task $task): bool
    {
        return $this->update($user, $task);
    }

    public function delete(User $user, Task $task): bool
    {
        $decision = AccessControl::overridePermissionState($user, 'tasks', 'delete');
        if ($decision !== null) {
            return $decision;
        }

        if (!AccessControl::roleAllows($user, 'tasks', 'delete')) {
            return false;
        }

        return AccessControl::isElevated($user) || $task->creator_id === $user->id;
    }

    private function isProjectMember(User $user, Task $task): bool
    {
        if (! $task->project_id) {
            return false;
        }

        return $task->project()
            ->whereHas('members', fn ($query) => $query->where('users.id', $user->id))
            ->exists();
    }

}
