<?php

namespace App\Http\Controllers;

use App\Http\Resources\HrRequestResource;
use App\Http\Resources\HrSettingResource;
use App\Models\HrRequest;
use App\Models\HrSetting;
use App\Models\User;
use App\Support\AccessControl;
use App\Support\CrmModuleManager;
use App\Support\SectionAccessManager;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class HrController extends Controller
{
    public function index(Request $request, SectionAccessManager $sectionAccessManager): View|JsonResponse
    {
        $this->authorize('viewAny', HrRequest::class);

        $user = $request->user();
        $isElevated = AccessControl::isElevated($user);

        $search = trim((string) $request->input('q', ''));
        $type = trim((string) $request->input('type', ''));
        $status = trim((string) $request->input('status', ''));
        $priority = trim((string) $request->input('priority', ''));
        $dateFrom = trim((string) $request->input('date_from', ''));
        $dateTo = trim((string) $request->input('date_to', ''));
        $selectedUserId = $isElevated ? $request->integer('user_id') : (int) $user->id;

        $requests = HrRequest::query()
            ->with(['employee:id,name,email,job_title', 'assignee:id,name,email,job_title', 'creator:id,name,email'])
            ->when($search !== '', function ($query) use ($search): void {
                $query->where(function ($sub) use ($search): void {
                    $sub->where('title', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%")
                        ->orWhere('external_id', 'like', "%{$search}%");
                });
            })
            ->when(array_key_exists($type, $this->typeOptions()), fn ($query) => $query->where('type', $type))
            ->when(array_key_exists($status, $this->statusOptions()), fn ($query) => $query->where('status', $status))
            ->when(array_key_exists($priority, $this->priorityOptions()), fn ($query) => $query->where('priority', $priority))
            ->when($dateFrom !== '', fn ($query) => $query->whereDate('requested_date', '>=', $dateFrom))
            ->when($dateTo !== '', fn ($query) => $query->whereDate('requested_date', '<=', $dateTo))
            ->when(! $isElevated, function ($query) use ($user): void {
                $query->where(function ($sub) use ($user): void {
                    $sub->where('employee_user_id', $user->id)
                        ->orWhere('assignee_user_id', $user->id)
                        ->orWhere('creator_user_id', $user->id);
                });
            })
            ->when($isElevated && $selectedUserId > 0, function ($query) use ($selectedUserId): void {
                $query->where(function ($sub) use ($selectedUserId): void {
                    $sub->where('employee_user_id', $selectedUserId)
                        ->orWhere('assignee_user_id', $selectedUserId)
                        ->orWhere('creator_user_id', $selectedUserId);
                });
            })
            ->orderByRaw('due_date is null')
            ->orderBy('due_date')
            ->orderByDesc('created_at')
            ->paginate(25)
            ->withQueryString();

        $settings = HrSetting::query()->first();
        if (! $settings) {
            $settings = new HrSetting([
                'provider' => 'internal',
                'is_active' => true,
                'default_sla_days' => 3,
                'allow_employee_requests' => true,
                'require_approval' => true,
                'webhook_secret' => Str::random(40),
            ]);
        } elseif (! $settings->webhook_secret) {
            $settings->webhook_secret = Str::random(40);
        }

        $canManageSettings = AccessControl::allows($user, 'hr', 'update');
        $canCreateRequest = AccessControl::allows($user, 'hr', 'create');
        $canUpdateRequest = AccessControl::allows($user, 'hr', 'update');
        $canDeleteRequest = AccessControl::allows($user, 'hr', 'delete');

        $canManageSectionAccess = $sectionAccessManager->canManage($user, 'hr');
        $sectionAccessUsers = $canManageSectionAccess
            ? User::query()->orderBy('name')->get(['id', 'name', 'email', 'role', 'permissions'])
            : collect();

        $userOptions = $isElevated
            ? User::query()->orderBy('name')->get(['id', 'name', 'email', 'job_title'])
            : User::query()->whereKey($user->id)->get(['id', 'name', 'email', 'job_title']);

        $requestPayload = HrRequestResource::collection($requests)->response()->getData(true);
        $settingsPayload = HrSettingResource::make($settings)->resolve();

        $payload = [
            'filters' => [
                'q' => $search,
                'type' => $type,
                'status' => $status,
                'priority' => $priority,
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
                'user_id' => $selectedUserId > 0 ? $selectedUserId : null,
            ],
            'options' => [
                'types' => $this->typeOptions(),
                'statuses' => $this->statusOptions(),
                'priorities' => $this->priorityOptions(),
                'users' => $this->userOptionsPayload($userOptions),
            ],
            'requests' => $requestPayload['data'] ?? [],
            'pagination' => $requestPayload['meta'] ?? [],
            'settings' => $settingsPayload,
            'permissions' => [
                'is_elevated' => $isElevated,
                'can_manage_settings' => $canManageSettings,
                'can_create_request' => $canCreateRequest,
                'can_update_request' => $canUpdateRequest,
                'can_delete_request' => $canDeleteRequest,
            ],
            'urls' => [
                'index' => route('hr.index'),
                'settings_update' => route('hr.settings.update'),
                'request_store' => route('hr.requests.store'),
                'request_update_template' => route('hr.requests.update', ['hrRequest' => '__ID__']),
                'request_delete_template' => route('hr.requests.destroy', ['hrRequest' => '__ID__']),
                'webhook' => route('hr.webhook'),
            ],
        ];

        if ($this->wantsJsonResponse($request)) {
            return response()->json($payload);
        }

        return view('hr.index', [
            'hrPayload' => $payload,
            'canManageSectionAccess' => $canManageSectionAccess,
            'sectionAccessUsers' => $sectionAccessUsers,
        ]);
    }

    public function updateSettings(Request $request): RedirectResponse|JsonResponse
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'hr', 'update'), 403);

        $validated = $request->validate([
            'provider' => ['required', 'string', 'max:60'],
            'is_active' => ['nullable', 'boolean'],
            'api_base_url' => ['nullable', 'string', 'max:255'],
            'api_key' => ['nullable', 'string', 'max:255'],
            'api_secret' => ['nullable', 'string', 'max:255'],
            'webhook_secret' => ['nullable', 'string', 'max:255'],
            'auto_assign_user_id' => ['nullable', 'integer', 'exists:users,id'],
            'default_sla_days' => ['required', 'integer', 'min:1', 'max:365'],
            'allow_employee_requests' => ['nullable', 'boolean'],
            'require_approval' => ['nullable', 'boolean'],
            'meta' => ['nullable', 'json'],
        ]);

        $payload = [
            'provider' => trim((string) $validated['provider']),
            'is_active' => $request->boolean('is_active'),
            'api_base_url' => $this->nullableText($validated['api_base_url'] ?? null),
            'api_key' => $this->nullableText($validated['api_key'] ?? null),
            'api_secret' => $this->nullableText($validated['api_secret'] ?? null),
            'webhook_secret' => $this->nullableText($validated['webhook_secret'] ?? null),
            'auto_assign_user_id' => $validated['auto_assign_user_id'] ?? null,
            'default_sla_days' => (int) $validated['default_sla_days'],
            'allow_employee_requests' => $request->boolean('allow_employee_requests', true),
            'require_approval' => $request->boolean('require_approval', true),
            'meta' => $this->decodeJsonArray($validated['meta'] ?? null),
        ];

        if (($payload['webhook_secret'] ?? '') === '') {
            $payload['webhook_secret'] = HrSetting::query()->value('webhook_secret') ?: Str::random(40);
        }

        $settings = HrSetting::query()->first() ?? new HrSetting();
        $settings->fill($payload)->save();

        if ($this->wantsJsonResponse($request)) {
            return response()->json([
                'message' => __('HR settings have been updated.'),
                'settings' => HrSettingResource::make($settings->loadMissing('autoAssignUser'))->resolve(),
            ]);
        }

        return back()->with('success', __('HR settings have been updated.'));
    }

    public function storeRequest(Request $request, CrmModuleManager $moduleManager): RedirectResponse|JsonResponse
    {
        $this->authorize('create', HrRequest::class);

        $payload = $this->validatedRequestData($request);
        $actor = $request->user();
        $isElevated = AccessControl::isElevated($actor);

        if (! $isElevated) {
            if (($payload['employee_user_id'] ?? null) !== null && (int) $payload['employee_user_id'] !== (int) $actor->id) {
                abort(403);
            }

            $payload['employee_user_id'] = $actor->id;
            $payload['creator_user_id'] = $actor->id;
            $payload['assignee_user_id'] = null;
        } elseif (! ($payload['creator_user_id'] ?? null)) {
            $payload['creator_user_id'] = $actor->id;
        }

        $settings = HrSetting::query()->first();
        if (! $isElevated && $settings && ! ($settings->allow_employee_requests ?? true)) {
            abort(403);
        }

        if (($payload['assignee_user_id'] ?? null) === null && ($settings?->auto_assign_user_id ?? null)) {
            $payload['assignee_user_id'] = (int) $settings->auto_assign_user_id;
        }

        $payload['requested_date'] = $payload['requested_date'] ?: now()->toDateString();
        if (($payload['due_date'] ?? null) === null && ($settings?->default_sla_days ?? 0) > 0) {
            $payload['due_date'] = now()->addDays((int) $settings->default_sla_days)->toDateString();
        }
        if (($payload['status'] ?? '') === 'resolved' && ($payload['resolved_date'] ?? null) === null) {
            $payload['resolved_date'] = now()->toDateString();
        }

        $payload = $moduleManager->applyPayloadHooks('hr.requests.store', $payload, [
            'hook' => 'hr.requests.store',
            'source' => 'web',
            'user_id' => $actor->id,
        ], array_keys($payload));

        $hrRequest = HrRequest::query()->create($payload);

        if ($this->wantsJsonResponse($request)) {
            return response()->json([
                'message' => __('HR request has been created.'),
                'request' => HrRequestResource::make($hrRequest->load(['employee', 'assignee', 'creator']))->resolve(),
            ], 201);
        }

        return redirect()
            ->route('hr.index')
            ->with('success', __('HR request has been created.'));
    }

    public function updateRequest(Request $request, HrRequest $hrRequest, CrmModuleManager $moduleManager): RedirectResponse|JsonResponse
    {
        $this->authorize('update', $hrRequest);

        $payload = $this->validatedRequestData($request, $hrRequest);
        $actor = $request->user();

        if (! AccessControl::isElevated($actor)) {
            $payload['employee_user_id'] = $hrRequest->employee_user_id;
            $payload['assignee_user_id'] = $hrRequest->assignee_user_id;
            $payload['creator_user_id'] = $hrRequest->creator_user_id ?: $actor->id;
            $payload['external_id'] = $hrRequest->external_id;
        } elseif (! ($payload['creator_user_id'] ?? null)) {
            $payload['creator_user_id'] = $hrRequest->creator_user_id ?: $actor->id;
        }

        if (($payload['status'] ?? '') === 'resolved' && ($payload['resolved_date'] ?? null) === null) {
            $payload['resolved_date'] = now()->toDateString();
        }
        if (($payload['status'] ?? '') !== 'resolved') {
            $payload['resolved_date'] = null;
        }

        $payload = $moduleManager->applyPayloadHooks('hr.requests.update', $payload, [
            'hook' => 'hr.requests.update',
            'source' => 'web',
            'user_id' => $actor->id,
            'hr_request_id' => $hrRequest->id,
        ], array_keys($payload));

        $hrRequest->fill($payload)->save();

        if ($this->wantsJsonResponse($request)) {
            return response()->json([
                'message' => __('HR request has been updated.'),
                'request' => HrRequestResource::make($hrRequest->fresh()->load(['employee', 'assignee', 'creator']))->resolve(),
            ]);
        }

        return back()->with('success', __('HR request has been updated.'));
    }

    public function destroyRequest(Request $request, HrRequest $hrRequest): RedirectResponse|JsonResponse
    {
        $this->authorize('delete', $hrRequest);

        $hrRequest->delete();

        if ($this->wantsJsonResponse($request)) {
            return response()->json([
                'message' => __('HR request has been deleted.'),
            ]);
        }

        return back()->with('success', __('HR request has been deleted.'));
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedRequestData(Request $request, ?HrRequest $hrRequest = null): array
    {
        $validated = $request->validate([
            'external_id' => [
                'nullable',
                'string',
                'max:255',
                Rule::unique('hr_requests', 'external_id')->ignore($hrRequest?->id),
            ],
            'type' => ['required', Rule::in(array_keys($this->typeOptions()))],
            'status' => ['required', Rule::in(array_keys($this->statusOptions()))],
            'priority' => ['required', Rule::in(array_keys($this->priorityOptions()))],
            'title' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'requested_date' => ['nullable', 'date'],
            'due_date' => ['nullable', 'date'],
            'resolved_date' => ['nullable', 'date'],
            'employee_user_id' => ['nullable', 'integer', 'exists:users,id'],
            'assignee_user_id' => ['nullable', 'integer', 'exists:users,id'],
            'creator_user_id' => ['nullable', 'integer', 'exists:users,id'],
            'meta' => ['nullable', 'json'],
        ]);

        return [
            'external_id' => $this->nullableText($validated['external_id'] ?? null),
            'type' => (string) $validated['type'],
            'status' => (string) $validated['status'],
            'priority' => (string) $validated['priority'],
            'title' => trim((string) $validated['title']),
            'description' => $this->nullableText($validated['description'] ?? null),
            'requested_date' => $validated['requested_date'] ?? null,
            'due_date' => $validated['due_date'] ?? null,
            'resolved_date' => $validated['resolved_date'] ?? null,
            'employee_user_id' => $validated['employee_user_id'] ?? null,
            'assignee_user_id' => $validated['assignee_user_id'] ?? null,
            'creator_user_id' => $validated['creator_user_id'] ?? null,
            'meta' => $this->decodeJsonArray($validated['meta'] ?? null),
        ];
    }

    /**
     * @return array<string, string>
     */
    private function typeOptions(): array
    {
        return [
            'vacation' => __('Vacation'),
            'sick_leave' => __('Sick leave'),
            'business_trip' => __('Business trip'),
            'reimbursement' => __('Reimbursement'),
            'onboarding' => __('Onboarding'),
            'offboarding' => __('Offboarding'),
            'other' => __('Other'),
        ];
    }

    /**
     * @return array<string, string>
     */
    private function statusOptions(): array
    {
        return [
            'new' => __('New'),
            'in_progress' => __('In progress'),
            'waiting_approval' => __('Waiting approval'),
            'resolved' => __('Resolved'),
            'rejected' => __('Rejected'),
        ];
    }

    /**
     * @return array<string, string>
     */
    private function priorityOptions(): array
    {
        return [
            'low' => __('Low'),
            'medium' => __('Medium'),
            'high' => __('High'),
            'urgent' => __('Urgent'),
        ];
    }

    /**
     * @param  Collection<int, User>  $users
     * @return list<array{id:int,name:string,email:string,job_title:string}>
     */
    private function userOptionsPayload(Collection $users): array
    {
        return $users->map(static function (User $member): array {
            return [
                'id' => (int) $member->id,
                'name' => (string) $member->name,
                'email' => (string) ($member->email ?? ''),
                'job_title' => (string) ($member->job_title ?? ''),
            ];
        })->values()->all();
    }

    private function nullableText(mixed $value): ?string
    {
        $value = trim((string) $value);

        return $value === '' ? null : $value;
    }

    /**
     * @return array<string, mixed>|null
     */
    private function decodeJsonArray(mixed $value): ?array
    {
        $raw = trim((string) $value);
        if ($raw === '') {
            return null;
        }

        /** @var mixed $decoded */
        $decoded = json_decode($raw, true);

        return is_array($decoded) ? $decoded : null;
    }

    private function wantsJsonResponse(Request $request): bool
    {
        return $request->expectsJson()
            || $request->wantsJson()
            || $request->isXmlHttpRequest()
            || $request->boolean('ajax');
    }
}
