<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\OneCSettingResource;
use App\Models\OneCSetting;
use App\Support\AccessControl;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class OneCSettingsController extends Controller
{
    public function show(Request $request): OneCSettingResource
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'onec', 'read'), 403);

        $settings = OneCSetting::query()->first();
        if (! $settings) {
            $settings = new OneCSetting([
                'provider' => '1c',
                'is_active' => false,
                'exchange_mode' => 'bidirectional',
                'auto_sync' => false,
                'sync_tasks' => true,
                'sync_deals' => true,
                'sync_companies' => true,
                'sync_contacts' => true,
                'sync_products' => true,
                'sync_warehouses' => true,
                'sync_interval_minutes' => 15,
                'webhook_secret' => Str::random(40),
            ]);
        }

        return OneCSettingResource::make($settings);
    }

    public function update(Request $request): OneCSettingResource
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'onec', 'update'), 403);

        $validated = $request->validate([
            'provider' => ['required', 'string', 'max:60'],
            'is_active' => ['nullable', 'boolean'],
            'endpoint_url' => ['nullable', 'string', 'max:255'],
            'username' => ['nullable', 'string', 'max:255'],
            'password' => ['nullable', 'string', 'max:255'],
            'api_key' => ['nullable', 'string', 'max:255'],
            'api_secret' => ['nullable', 'string', 'max:255'],
            'webhook_secret' => ['nullable', 'string', 'max:255'],
            'exchange_mode' => ['required', Rule::in($this->exchangeModes())],
            'auto_sync' => ['nullable', 'boolean'],
            'sync_tasks' => ['nullable', 'boolean'],
            'sync_deals' => ['nullable', 'boolean'],
            'sync_companies' => ['nullable', 'boolean'],
            'sync_contacts' => ['nullable', 'boolean'],
            'sync_products' => ['nullable', 'boolean'],
            'sync_warehouses' => ['nullable', 'boolean'],
            'sync_interval_minutes' => ['required', 'integer', 'min:1', 'max:1440'],
            'meta' => ['nullable', 'array'],
        ]);

        $payload = [
            'provider' => trim((string) $validated['provider']),
            'is_active' => $request->boolean('is_active'),
            'endpoint_url' => $this->nullableText($validated['endpoint_url'] ?? null),
            'username' => $this->nullableText($validated['username'] ?? null),
            'password' => $this->nullableText($validated['password'] ?? null),
            'api_key' => $this->nullableText($validated['api_key'] ?? null),
            'api_secret' => $this->nullableText($validated['api_secret'] ?? null),
            'webhook_secret' => $this->nullableText($validated['webhook_secret'] ?? null),
            'exchange_mode' => (string) $validated['exchange_mode'],
            'auto_sync' => $request->boolean('auto_sync'),
            'sync_tasks' => $request->boolean('sync_tasks'),
            'sync_deals' => $request->boolean('sync_deals'),
            'sync_companies' => $request->boolean('sync_companies'),
            'sync_contacts' => $request->boolean('sync_contacts'),
            'sync_products' => $request->boolean('sync_products'),
            'sync_warehouses' => $request->boolean('sync_warehouses'),
            'sync_interval_minutes' => (int) $validated['sync_interval_minutes'],
            'meta' => is_array($validated['meta'] ?? null) ? $validated['meta'] : null,
        ];

        if (($payload['webhook_secret'] ?? '') === '') {
            $payload['webhook_secret'] = OneCSetting::query()->value('webhook_secret') ?: Str::random(40);
        }

        $settings = OneCSetting::query()->first() ?? new OneCSetting();
        $settings->fill($payload)->save();

        return OneCSettingResource::make($settings);
    }

    /**
     * @return list<string>
     */
    private function exchangeModes(): array
    {
        return ['inbound', 'outbound', 'bidirectional'];
    }

    private function nullableText(mixed $value): ?string
    {
        $value = trim((string) $value);

        return $value === '' ? null : $value;
    }
}

