<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseAddress;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class WarehouseSectionTest extends TestCase
{
    use RefreshDatabase;

    public function test_manager_can_open_warehouses_index_and_create_warehouse(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $this->actingAs($manager)
            ->get(route('warehouses.index'))
            ->assertOk()
            ->assertSee('Warehouses');

        $this->actingAs($manager)
            ->post(route('warehouses.store'), [
                'name' => 'Main storage',
                'code' => 'WH-MAIN-1',
                'location' => 'Central district',
                'capacity' => 2500,
                'status' => 'active',
                'manager_id' => $manager->id,
            ])
            ->assertRedirect();

        $this->assertDatabaseHas('warehouses', [
            'name' => 'Main storage',
            'code' => 'WH-MAIN-1',
            'status' => 'active',
        ]);
    }

    public function test_regular_user_cannot_open_warehouses_section_by_default(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
        ]);

        $this->actingAs($user)
            ->get(route('warehouses.index'))
            ->assertForbidden();
    }

    public function test_personalized_permissions_allow_regular_user_to_manage_warehouses(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'warehouses.read' => true,
                'warehouses.create' => true,
                'warehouses.update' => true,
                'warehouses.delete' => true,
            ],
        ]);

        $this->actingAs($user)
            ->post(route('warehouses.store'), [
                'name' => 'Custom warehouse',
                'code' => 'WH-CUSTOM-1',
                'location' => 'North district',
                'capacity' => 1500,
                'status' => 'active',
                'manager_id' => $user->id,
            ])
            ->assertRedirect();

        $warehouse = Warehouse::query()->where('code', 'WH-CUSTOM-1')->firstOrFail();

        $this->actingAs($user)
            ->put(route('warehouses.update', $warehouse), [
                'name' => 'Custom warehouse updated',
                'code' => 'WH-CUSTOM-1',
                'location' => 'North district',
                'capacity' => 1400,
                'status' => 'inactive',
                'manager_id' => $user->id,
            ])
            ->assertRedirect(route('warehouses.show', $warehouse));

        $this->assertDatabaseHas('warehouses', [
            'id' => $warehouse->id,
            'name' => 'Custom warehouse updated',
            'status' => 'inactive',
        ]);
    }

    public function test_manager_can_generate_warehouse_map_and_manage_addresses(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $warehouse = Warehouse::factory()->create([
            'manager_id' => $manager->id,
            'map_rows' => 3,
            'map_columns' => 4,
        ]);

        $this->actingAs($manager)
            ->patch(route('warehouses.map.update', $warehouse), [
                'map_rows' => 4,
                'map_columns' => 5,
            ])
            ->assertRedirect(route('warehouses.show', $warehouse));

        $warehouse->refresh();
        $this->assertSame(4, (int) $warehouse->map_rows);
        $this->assertSame(5, (int) $warehouse->map_columns);

        $this->actingAs($manager)
            ->post(route('warehouses.addresses.generate', $warehouse), [
                'prefix' => 'A',
                'zone' => 'Main',
                'color' => '#10B981',
                'clear_existing' => 1,
            ])
            ->assertRedirect(route('warehouses.show', $warehouse));

        $this->assertDatabaseHas('warehouse_addresses', [
            'warehouse_id' => $warehouse->id,
            'code' => 'A-01-01',
            'x' => 1,
            'y' => 1,
            'zone' => 'Main',
        ]);

        $address = WarehouseAddress::query()
            ->where('warehouse_id', $warehouse->id)
            ->where('code', 'A-01-01')
            ->firstOrFail();

        $this->actingAs($manager)
            ->patch(route('warehouses.addresses.update', [$warehouse, $address]), [
                'code' => 'A-01-01',
                'zone' => 'Main',
                'aisle' => '1',
                'rack' => '1',
                'shelf' => '2',
                'cell' => '7',
                'x' => 1,
                'y' => 1,
                'color' => '#F97316',
                'capacity' => 100,
                'current_load' => 25,
                'status' => 'reserved',
                'note' => 'Reserved slot',
            ])
            ->assertRedirect(route('warehouses.show', $warehouse));

        $this->assertDatabaseHas('warehouse_addresses', [
            'id' => $address->id,
            'status' => 'reserved',
            'shelf' => '2',
            'cell' => '7',
        ]);

        $this->actingAs($manager)
            ->delete(route('warehouses.addresses.destroy', [$warehouse, $address]))
            ->assertRedirect(route('warehouses.show', $warehouse));

        $this->assertDatabaseMissing('warehouse_addresses', [
            'id' => $address->id,
        ]);
    }
}
