<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TaskWorkbenchTest extends TestCase
{
    use RefreshDatabase;

    public function test_task_pages_support_kanban_and_gantt_views(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        Task::factory()->count(3)->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'starts_at' => now()->subDays(2),
            'due_at' => now()->addDays(5),
        ]);

        $this->actingAs($user)
            ->get(route('tasks.index', ['view' => 'kanban']))
            ->assertOk()
            ->assertSee('Drag-and-drop', false);

        $this->actingAs($user)
            ->get(route('tasks.index', ['view' => 'gantt']))
            ->assertOk()
            ->assertSee('Period', false);
    }

    public function test_task_filter_shows_compact_defaults_and_collapsed_advanced_section(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $this->actingAs($user)
            ->get(route('tasks.index'))
            ->assertOk()
            ->assertSee('Title contains')
            ->assertSee('id="status"', false)
            ->assertSee('id="assignee_id"', false)
            ->assertSee('More filters')
            ->assertSee('data-advanced-task-filters', false)
            ->assertSee('data-advanced-open="0"', false);
    }

    public function test_task_filter_search_uses_title_contains_and_opens_advanced_block_when_needed(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $titleTask = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'title' => 'Alpha release',
            'description' => 'Unrelated notes',
            'priority' => 'high',
        ]);

        $descriptionOnlyTask = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'title' => 'Roadmap task',
            'description' => 'Contains Alpha in description only',
            'priority' => 'medium',
        ]);

        $this->actingAs($user)
            ->get(route('tasks.index', [
                'q' => 'Alpha',
                'priority' => 'high',
            ]))
            ->assertOk()
            ->assertSee($titleTask->title)
            ->assertDontSee($descriptionOnlyTask->title)
            ->assertSee('data-advanced-open="1"', false);
    }

    public function test_task_view_selection_is_saved_and_used_as_default(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->get(route('tasks.index', ['view' => 'kanban']))
            ->assertOk()
            ->assertSee('id="task-kanban"', false);

        $this->assertDatabaseHas('users', [
            'id' => $user->id,
            'preferred_task_view' => 'kanban',
        ]);

        $this->actingAs($user)
            ->get(route('tasks.index'))
            ->assertOk()
            ->assertSee('id="task-kanban"', false);
    }

    public function test_task_status_can_be_updated_from_kanban_ajax(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
        ]);

        $this->actingAs($user)
            ->patchJson(route('tasks.update-status', $task), [
                'status' => 'in_progress',
            ])
            ->assertOk()
            ->assertJsonPath('task.id', $task->id)
            ->assertJsonPath('task.status', 'in_progress');

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'status' => 'in_progress',
        ]);
    }

    public function test_subtask_inherits_parent_project_when_not_provided(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $user->id,
            'manager_id' => $user->id,
            'status' => 'active',
        ]);

        $parent = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'project_id' => $project->id,
        ]);

        $this->actingAs($user)
            ->post(route('tasks.store'), [
                'title' => 'Child task',
                'parent_id' => $parent->id,
                'status' => 'todo',
                'priority' => 'medium',
                'estimated_hours' => 4,
                'tracked_hours' => 0,
            ])
            ->assertRedirect();

        $this->assertDatabaseHas('tasks', [
            'title' => 'Child task',
            'parent_id' => $parent->id,
            'project_id' => $project->id,
        ]);
    }

    public function test_user_can_store_personal_task_kanban_preset(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $response = $this->actingAs($user)
            ->patch(route('tasks.kanban.preset'), [
                'labels' => [
                    'todo' => 'Plan',
                    'in_progress' => 'In progress',
                    'review' => 'Agreement',
                    'done' => 'Ready',
                ],
                'sort' => [
                    'todo' => 2,
                    'in_progress' => 1,
                    'review' => 3,
                    'done' => 4,
                ],
            ]);

        $response->assertRedirect();
        $response->assertSessionHasNoErrors();

        $user->refresh();

        $this->assertIsArray($user->task_kanban_preset);
        $this->assertSame('in_progress', $user->task_kanban_preset[0]['status'] ?? null);
        $this->assertSame('In progress', $user->task_kanban_preset[0]['label'] ?? null);

        $this->actingAs($user)
            ->get(route('tasks.index', ['view' => 'kanban']))
            ->assertOk()
            ->assertSeeInOrder(['In progress', 'Plan', 'Agreement', 'Ready'], false);
    }

    public function test_user_can_add_custom_kanban_stage_with_color_and_order(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $response = $this->actingAs($user)
            ->patch(route('tasks.kanban.preset'), [
                'stages' => [
                    ['status' => 'todo', 'label' => 'Plan', 'color' => '#94A3B8', 'is_core' => 1],
                    ['status' => 'custom_waiting_client', 'label' => 'Client waiting', 'color' => '#FF8800', 'is_core' => 0],
                    ['status' => 'in_progress', 'label' => 'In progress', 'color' => '#3B82F6', 'is_core' => 1],
                    ['status' => 'review', 'label' => 'Check', 'color' => '#F59E0B', 'is_core' => 1],
                    ['status' => 'done', 'label' => 'Done', 'color' => '#10B981', 'is_core' => 1],
                ],
            ]);

        $response->assertRedirect();
        $response->assertSessionHasNoErrors();

        $user->refresh();

        $this->assertIsArray($user->task_kanban_preset);
        $this->assertSame('custom_waiting_client', $user->task_kanban_preset[1]['status'] ?? null);
        $this->assertSame('#FF8800', $user->task_kanban_preset[1]['color'] ?? null);

        $this->actingAs($user)
            ->get(route('tasks.index', ['view' => 'kanban']))
            ->assertOk()
            ->assertSeeInOrder(['Plan', 'Client waiting', 'In progress', 'Check', 'Done'], false)
            ->assertSee('border-top-color: #FF8800;', false);
    }

    public function test_task_status_can_be_updated_to_custom_kanban_stage(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
            'task_kanban_preset' => [
                ['status' => 'todo', 'label' => 'Plan', 'color' => '#94A3B8'],
                ['status' => 'custom_waiting_client', 'label' => 'Client waiting', 'color' => '#FF8800'],
                ['status' => 'in_progress', 'label' => 'In progress', 'color' => '#3B82F6'],
                ['status' => 'review', 'label' => 'Check', 'color' => '#F59E0B'],
                ['status' => 'done', 'label' => 'Done', 'color' => '#10B981'],
            ],
        ]);

        $task = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
        ]);

        $this->actingAs($user)
            ->patchJson(route('tasks.update-status', $task), [
                'status' => 'custom_waiting_client',
            ])
            ->assertOk()
            ->assertJsonPath('task.id', $task->id)
            ->assertJsonPath('task.status', 'custom_waiting_client');

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'status' => 'custom_waiting_client',
        ]);
    }

    public function test_task_can_be_updated_via_ajax_live_editor_endpoint(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
            'priority' => 'medium',
        ]);

        $startsAt = now()->addDay()->setSecond(0);
        $dueAt = now()->addDays(3)->setSecond(0);

        $this->actingAs($user)
            ->patchJson(route('tasks.update', $task), [
                'title' => 'Live update title',
                'description' => 'Live update description',
                'status' => 'review',
                'priority' => 'high',
                'assignee_id' => $user->id,
                'starts_at' => $startsAt->toDateTimeString(),
                'due_at' => $dueAt->toDateTimeString(),
                'estimated_hours' => 5.5,
                'tracked_hours' => 2.25,
                'sort_order' => 7,
            ])
            ->assertOk()
            ->assertJsonPath('task.id', $task->id)
            ->assertJsonPath('task.title', 'Live update title')
            ->assertJsonPath('task.status', 'review')
            ->assertJsonPath('task.priority', 'high');

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'title' => 'Live update title',
            'status' => 'review',
            'priority' => 'high',
            'sort_order' => 7,
        ]);
    }

    public function test_task_show_and_sidebar_use_live_editor_without_edit_button(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->get(route('tasks.show', $task))
            ->assertOk()
            ->assertSee('data-task-live-form', false)
            ->assertDontSee(route('tasks.edit', $task), false);

        $this->actingAs($user)
            ->get(route('tasks.show', $task).'?sidepanel=1')
            ->assertOk()
            ->assertSee('data-task-live-form', false)
            ->assertDontSee(route('tasks.edit', $task), false);
    }
}
