<?php

namespace Tests\Feature;

use App\Models\Activity;
use App\Models\Task;
use App\Models\User;
use App\Models\WebForm;
use App\Support\RightRailManager;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class RightRailNotificationReadTest extends TestCase
{
    use RefreshDatabase;

    public function test_opening_notification_marks_it_as_viewed_and_hides_it_from_list(): void
    {
        $user = User::factory()->create(['role' => 'admin']);
        $activity = Activity::factory()->create([
            'user_id' => $user->id,
            'occurred_at' => now(),
        ]);

        $key = 'activity-'.$activity->id;
        $manager = app(RightRailManager::class);

        $initialNotificationKeys = $manager->data($user)['notifications']
            ->pluck('key')
            ->all();

        $this->assertContains($key, $initialNotificationKeys);

        $this->actingAs($user)
            ->get(route('notifications.open', ['key' => $key]))
            ->assertRedirect(route('deals.show', $activity->deal));

        $user->refresh();

        $this->assertContains($key, $user->right_rail_viewed_notifications ?? []);

        $afterOpenKeys = $manager->data($user)['notifications']
            ->pluck('key')
            ->all();

        $this->assertNotContains($key, $afterOpenKeys);
    }

    public function test_invalid_notification_key_falls_back_to_dashboard(): void
    {
        $user = User::factory()->create(['role' => 'admin']);

        $this->actingAs($user)
            ->get(route('notifications.open', ['key' => 'unknown-key']))
            ->assertRedirect(route('dashboard'));
    }

    public function test_notifications_feed_returns_latest_items_for_right_rail_polling(): void
    {
        $user = User::factory()->create(['role' => 'admin']);
        $activity = Activity::factory()->create([
            'user_id' => $user->id,
            'occurred_at' => now(),
        ]);

        $notificationKey = 'activity-'.$activity->id;

        $this->actingAs($user)
            ->getJson(route('notifications.feed'))
            ->assertOk()
            ->assertJsonPath('notifications.0.key', $notificationKey)
            ->assertJsonPath('notifications.0.open_url', route('notifications.open', ['key' => $notificationKey]))
            ->assertJsonPath('unread_count', 1);
    }

    public function test_assignee_receives_notification_when_new_task_is_created_for_him(): void
    {
        $creator = User::factory()->create(['role' => 'admin']);
        $assignee = User::factory()->create(['role' => 'admin']);

        $task = Task::query()->create([
            'title' => 'Prepare commercial proposal',
            'creator_id' => $creator->id,
            'assignee_id' => $assignee->id,
            'status' => 'todo',
            'priority' => 'medium',
        ]);

        $manager = app(RightRailManager::class);
        $assigneeNotifications = $manager->data($assignee)['notifications'];
        $creatorNotifications = $manager->data($creator)['notifications'];

        $taskNotification = $assigneeNotifications
            ->first(fn (array $item): bool => str_starts_with((string) ($item['key'] ?? ''), 'task-'.$task->id.'-assigned-'));

        $this->assertIsArray($taskNotification);
        $this->assertSame('New task assigned', $taskNotification['title'] ?? null);
        $this->assertSame(route('tasks.show', $task), $taskNotification['url'] ?? null);

        $creatorHasTaskNotification = $creatorNotifications
            ->contains(fn (array $item): bool => str_starts_with((string) ($item['key'] ?? ''), 'task-'.$task->id.'-assigned-'));

        $this->assertFalse($creatorHasTaskNotification);
    }

    public function test_form_owner_receives_submission_notification_with_executed_action(): void
    {
        $owner = User::factory()->create(['role' => 'admin']);

        $form = WebForm::query()->create([
            'owner_id' => $owner->id,
            'name' => 'Landing lead form',
            'slug' => 'landing-lead-form',
            'description' => 'Public form for website leads',
            'submit_label' => 'Send',
            'show_reset_button' => false,
            'reset_label' => 'Clear',
            'success_message' => 'Thanks',
            'is_active' => true,
            'behavior' => 'create_task',
            'behavior_settings' => [],
            'fields' => [
                [
                    'label' => 'Lead title',
                    'name' => 'lead_title',
                    'type' => 'text',
                    'required' => true,
                    'target' => 'title',
                    'placeholder' => '',
                    'options' => [],
                ],
            ],
        ]);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'lead_title' => 'Website lead #42',
        ])->assertRedirect(route('public-forms.show', ['form' => $form->slug]));

        $task = Task::query()->latest('id')->firstOrFail();
        $manager = app(RightRailManager::class);
        $notifications = $manager->data($owner)['notifications'];
        $submissionNotification = $notifications
            ->first(fn (array $item): bool => str_starts_with((string) ($item['key'] ?? ''), 'form-submission-'));

        $this->assertIsArray($submissionNotification);
        $this->assertSame('Form submitted', $submissionNotification['title'] ?? null);
        $this->assertStringContainsString('Landing lead form', (string) ($submissionNotification['subtitle'] ?? ''));
        $this->assertStringContainsString('Created task', (string) ($submissionNotification['subtitle'] ?? ''));

        $key = (string) ($submissionNotification['key'] ?? '');

        $this->actingAs($owner)
            ->get(route('notifications.open', ['key' => $key]))
            ->assertRedirect(route('tasks.show', $task));
    }
}
