<?php

namespace Tests\Feature;

use App\Models\Deal;
use App\Models\Pipeline;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class RbacPolicyTest extends TestCase
{
    use RefreshDatabase;

    public function test_sales_user_cannot_create_pipeline(): void
    {
        $sales = User::factory()->create([
            'role' => 'sales',
        ]);

        $this->actingAs($sales)
            ->post(route('pipelines.store'), [
                'name' => 'Forbidden pipeline',
                'sort_order' => 1,
                'is_active' => 1,
                'stages' => "New\nDone",
            ])
            ->assertForbidden();
    }

    public function test_manager_can_create_pipeline(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $this->actingAs($manager)
            ->post(route('pipelines.store'), [
                'name' => 'Manager pipeline',
                'sort_order' => 1,
                'is_active' => 1,
                'stages' => "New\nDone",
            ])
            ->assertRedirect();

        $this->assertDatabaseHas('pipelines', [
            'name' => 'Manager pipeline',
        ]);
    }

    public function test_update_stage_returns_json_for_kanban_ajax(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::create([
            'name' => 'Sales',
            'description' => null,
            'is_default' => true,
            'is_active' => true,
            'sort_order' => 0,
        ]);

        $stageA = $pipeline->stages()->create([
            'name' => 'Qualification',
            'code' => 'qualification_1',
            'sort_order' => 0,
            'probability' => 40,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $stageB = $pipeline->stages()->create([
            'name' => 'Negotiation',
            'code' => 'negotiation_2',
            'sort_order' => 1,
            'probability' => 70,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $deal = Deal::create([
            'title' => 'Ajax stage move',
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stageA->id,
            'owner_id' => $manager->id,
            'amount' => 12000,
            'currency' => 'USD',
            'priority' => 'medium',
            'status' => 'open',
            'source' => 'manual',
        ]);

        $this->actingAs($manager)
            ->patchJson(route('deals.update-stage', $deal), [
                'stage_id' => $stageB->id,
            ])
            ->assertOk()
            ->assertJsonPath('deal.id', $deal->id)
            ->assertJsonPath('deal.stage_id', $stageB->id);
    }

    public function test_update_stage_still_succeeds_when_broadcast_transport_is_unavailable(): void
    {
        config([
            'broadcasting.default' => 'reverb',
            'broadcasting.connections.reverb.key' => 'local',
            'broadcasting.connections.reverb.secret' => 'local',
            'broadcasting.connections.reverb.app_id' => 'local',
            'broadcasting.connections.reverb.options.host' => '127.0.0.1',
            'broadcasting.connections.reverb.options.port' => 9,
            'broadcasting.connections.reverb.options.scheme' => 'http',
            'broadcasting.connections.reverb.options.useTLS' => false,
        ]);

        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::create([
            'name' => 'Sales with failing broadcaster',
            'description' => null,
            'is_default' => true,
            'is_active' => true,
            'sort_order' => 0,
        ]);

        $stageA = $pipeline->stages()->create([
            'name' => 'Qualification',
            'code' => 'qualification_failover_1',
            'sort_order' => 0,
            'probability' => 40,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $stageB = $pipeline->stages()->create([
            'name' => 'Negotiation',
            'code' => 'negotiation_failover_2',
            'sort_order' => 1,
            'probability' => 70,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $deal = Deal::create([
            'title' => 'Broadcast failover move',
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stageA->id,
            'owner_id' => $manager->id,
            'amount' => 9000,
            'currency' => 'USD',
            'priority' => 'medium',
            'status' => 'open',
            'source' => 'manual',
        ]);

        $this->actingAs($manager)
            ->patchJson(route('deals.update-stage', $deal), [
                'stage_id' => $stageB->id,
            ])
            ->assertOk()
            ->assertJsonPath('deal.stage_id', $stageB->id);

        $this->assertDatabaseHas('deals', [
            'id' => $deal->id,
            'stage_id' => $stageB->id,
        ]);
    }
}
