<?php

namespace Tests\Feature;

use App\Models\UpdateCheck;
use App\Models\UpdateSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;

class ProfileUpdatesSettingsTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_can_open_updates_settings_tab_with_default_source(): void
    {
        $admin = User::factory()->create([
            'role' => 'admin',
        ]);

        $this->actingAs($admin)
            ->get(route('profile.edit', ['section' => 'updates']))
            ->assertOk()
            ->assertSee('Updates')
            ->assertSee('update.crm25.webnet.kz');
    }

    public function test_admin_can_update_update_settings(): void
    {
        $admin = User::factory()->create([
            'role' => 'admin',
        ]);

        $this->actingAs($admin)
            ->patch(route('profile.updates.update'), [
                'feed_url' => 'update.crm25.webnet.kz',
                'current_version' => '25.1.0',
                'channel' => 'beta',
                'is_active' => 1,
                'auto_check_enabled' => 1,
                'check_interval_minutes' => 360,
                'request_timeout_seconds' => 10,
                'verify_tls' => 1,
                '_settings_section' => 'updates',
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'updates']))
            ->assertSessionHasNoErrors();

        $this->assertDatabaseHas('update_settings', [
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.1.0',
            'channel' => 'beta',
            'check_interval_minutes' => 360,
            'request_timeout_seconds' => 10,
        ]);
    }

    public function test_admin_can_run_update_check_and_store_result(): void
    {
        $admin = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.0.0',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        Http::fake([
            'https://update.crm25.webnet.kz/api/v1/crm25/updates*' => Http::response([
                'version' => '25.2.0',
                'build' => '250200',
                'is_update_available' => true,
                'notes' => 'New release available.',
            ], 200),
            '*' => Http::response(['message' => 'not used'], 404),
        ]);

        $this->actingAs($admin)
            ->post(route('profile.updates.check'))
            ->assertRedirect(route('profile.edit', ['section' => 'updates']))
            ->assertSessionHasNoErrors();

        $this->assertDatabaseHas('update_checks', [
            'status' => 'success',
            'remote_version' => '25.2.0',
            'is_update_available' => true,
        ]);

        $check = UpdateCheck::query()->latest('id')->first();
        $this->assertNotNull($check);
        $this->assertSame('https://update.crm25.webnet.kz/api/v1/crm25/updates', $check?->endpoint_url);
    }

    public function test_non_admin_cannot_manage_update_settings(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
        ]);

        $this->actingAs($user)
            ->patch(route('profile.updates.update'), [
                'feed_url' => 'update.crm25.webnet.kz',
                'channel' => 'stable',
                'check_interval_minutes' => 720,
                'request_timeout_seconds' => 8,
            ])
            ->assertForbidden();

        $this->actingAs($user)
            ->post(route('profile.updates.check'))
            ->assertForbidden();
    }
}

