<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class ProfileTest extends TestCase
{
    use RefreshDatabase;

    public function test_profile_page_is_displayed(): void
    {
        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->get('/profile');

        $response->assertOk()
            ->assertSee('Profile')
            ->assertSee('Localization')
            ->assertSee('API')
            ->assertDontSee('API integration keys')
            ->assertDontSee('Menu settings')
            ->assertDontSee('Profile icon');
    }

    public function test_profile_settings_tab_can_be_opened_in_window_mode(): void
    {
        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->get('/profile?section=api');

        $response->assertOk()
            ->assertSee('API')
            ->assertSee('API integration keys')
            ->assertDontSee('Delete Account');
    }

    public function test_profile_update_keeps_window_context(): void
    {
        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->patch('/profile?section=localization', [
                'name' => $user->name,
                'email' => $user->email,
                'locale' => 'en',
                '_settings_section' => 'localization',
            ]);

        $response
            ->assertSessionHasNoErrors()
            ->assertRedirect('/profile?section=localization');
    }

    public function test_profile_information_can_be_updated(): void
    {
        $manager = User::factory()->create([
            'name' => 'Chief Executive',
        ]);

        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->patch('/profile', [
                'name' => 'Test User',
                'email' => 'test@example.com',
                'phone' => '+7 999 123-45-67',
                'job_title' => 'Account manager',
                'manager_id' => $manager->id,
                'birth_date' => '1994-07-20',
            ]);

        $response
            ->assertSessionHasNoErrors()
            ->assertRedirect('/profile');

        $user->refresh();

        $this->assertSame('Test User', $user->name);
        $this->assertSame('test@example.com', $user->email);
        $this->assertSame('+7 999 123-45-67', $user->phone);
        $this->assertSame('Account manager', $user->job_title);
        $this->assertSame($manager->id, $user->manager_id);
        $this->assertSame('1994-07-20', $user->birth_date?->toDateString());
        $this->assertNull($user->email_verified_at);
    }

    public function test_profile_form_contains_manager_selector_options(): void
    {
        $user = User::factory()->create();
        $firstManager = User::factory()->create(['name' => 'Anna Leader']);
        $secondManager = User::factory()->create(['name' => 'Ivan Director']);

        $this->actingAs($user)
            ->get('/profile')
            ->assertOk()
            ->assertSee('Supervisor')
            ->assertSee($firstManager->name)
            ->assertSee($secondManager->name);
    }

    public function test_user_cannot_assign_self_as_manager_in_profile(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->patch('/profile', [
                'name' => $user->name,
                'email' => $user->email,
                'manager_id' => $user->id,
            ])
            ->assertSessionHasErrors(['manager_id']);
    }

    public function test_email_verification_status_is_unchanged_when_the_email_address_is_unchanged(): void
    {
        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->patch('/profile', [
                'name' => 'Test User',
                'email' => $user->email,
            ]);

        $response
            ->assertSessionHasNoErrors()
            ->assertRedirect('/profile');

        $this->assertNotNull($user->refresh()->email_verified_at);
    }

    public function test_profile_photo_can_be_uploaded_with_zoom_and_centered_focus(): void
    {
        Storage::fake('public');
        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->patch('/profile', [
                'name' => $user->name,
                'email' => $user->email,
                'locale' => 'ru',
                'profile_photo' => UploadedFile::fake()->image('avatar.jpg', 800, 800),
                'profile_photo_zoom' => 1.55,
            ]);

        $response
            ->assertSessionHasNoErrors()
            ->assertRedirect('/profile');

        $user->refresh();

        $this->assertNotNull($user->profile_photo_path);
        $this->assertSame(50, $user->profile_photo_focus_x);
        $this->assertSame(50, $user->profile_photo_focus_y);
        $this->assertEquals(1.55, (float) $user->profile_photo_zoom);

        Storage::disk('public')->assertExists((string) $user->profile_photo_path);
    }

    public function test_uploading_new_profile_photo_keeps_center_and_updates_zoom(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'profile_photo_path' => 'profile-photos/current-avatar.jpg',
            'profile_photo_focus_x' => 18,
            'profile_photo_focus_y' => 74,
            'profile_photo_zoom' => 1.85,
        ]);

        Storage::disk('public')->put('profile-photos/current-avatar.jpg', 'old-photo');

        $response = $this
            ->actingAs($user)
            ->patch('/profile', [
                'name' => $user->name,
                'email' => $user->email,
                'profile_photo' => UploadedFile::fake()->image('new-avatar.jpg', 900, 900),
                'profile_photo_zoom' => 1.2,
            ]);

        $response
            ->assertSessionHasNoErrors()
            ->assertRedirect('/profile');

        $user->refresh();

        $this->assertNotNull($user->profile_photo_path);
        $this->assertNotSame('profile-photos/current-avatar.jpg', $user->profile_photo_path);
        $this->assertSame(50, $user->profile_photo_focus_x);
        $this->assertSame(50, $user->profile_photo_focus_y);
        $this->assertEquals(1.2, (float) $user->profile_photo_zoom);
    }

    public function test_user_can_delete_their_account(): void
    {
        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->delete('/profile', [
                'password' => 'password',
            ]);

        $response
            ->assertSessionHasNoErrors()
            ->assertRedirect('/');

        $this->assertGuest();
        $this->assertNull($user->fresh());
    }

    public function test_correct_password_must_be_provided_to_delete_account(): void
    {
        $user = User::factory()->create();

        $response = $this
            ->actingAs($user)
            ->from('/profile')
            ->delete('/profile', [
                'password' => 'wrong-password',
            ]);

        $response
            ->assertSessionHasErrorsIn('userDeletion', 'password')
            ->assertRedirect('/profile');

        $this->assertNotNull($user->fresh());
    }
}
