<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Laravel\Sanctum\PersonalAccessToken;
use Tests\TestCase;

class ProfileApiTokenTest extends TestCase
{
    use RefreshDatabase;

    public function test_api_settings_section_contains_api_documentation(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->get(route('profile.edit', ['section' => 'api']))
            ->assertOk()
            ->assertSeeText('API Documentation')
            ->assertSeeText('Full interactive documentation is placed on a separate page with a menu for entities.')
            ->assertSee(route('docs.api.page'), false)
            ->assertSee(route('docs.api'), false)
            ->assertSee(route('docs.modules.page'), false)
            ->assertSeeText('Open documentation page')
            ->assertSeeText('Modules documentation');
    }

    public function test_api_documentation_page_requires_authentication(): void
    {
        $this->get(route('docs.api.page'))
            ->assertRedirect(route('login'));
    }

    public function test_authenticated_user_can_open_api_documentation_page_with_entities_menu(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->get(route('docs.api.page'))
            ->assertOk()
            ->assertSeeText('API Documentation')
            ->assertSeeText('Entity menu')
            ->assertSeeText('Tasks')
            ->assertSeeText('Transactions')
            ->assertSeeText('Products')
            ->assertSeeText('Warehouses')
            ->assertSeeText('Companies')
            ->assertSeeText('Contacts')
            ->assertSeeText('Projects')
            ->assertSeeText('Disk')
            ->assertSeeText('HR Service')
            ->assertSeeText('Mail Service')
            ->assertSeeText('Request fields (POST/PATCH)')
            ->assertSeeText('Response fields')
            ->assertSeeText('Business restrictions')
            ->assertSeeText('after_or_equal:starts_at')
            ->assertSeeText('You cannot select a task for yourself in parent_id')
            ->assertSeeText('Token rights')
            ->assertSeeText('All endpoints');
    }

    public function test_api_markdown_documentation_requires_authentication(): void
    {
        $this->get(route('docs.api'))
            ->assertRedirect(route('login'));
    }

    public function test_authenticated_user_can_open_full_api_markdown_documentation(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->get(route('docs.api'))
            ->assertOk()
            ->assertSeeText('API CRM (v1)')
            ->assertSeeText('REST API');
    }

    public function test_user_can_issue_and_revoke_api_token_from_profile_settings(): void
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user)
            ->post(route('profile.api-tokens.store'), [
                'name' => 'Integration Key',
                'permissions' => [
                    'tasks' => ['read' => 1, 'create' => 1, 'update' => 1],
                    'deals' => ['read' => 1],
                    'products' => ['read' => 1],
                    'warehouses' => ['read' => 1],
                    'companies' => ['read' => 1],
                    'contacts' => ['read' => 1],
                    'projects' => ['read' => 1],
                    'disks' => ['read' => 1],
                ],
            ]);

        $response
            ->assertRedirect()
            ->assertSessionHasNoErrors()
            ->assertSessionHas('api_token_plain_text');

        $token = PersonalAccessToken::query()
            ->where('tokenable_id', $user->id)
            ->where('tokenable_type', User::class)
            ->first();

        $this->assertNotNull($token);
        $this->assertSame('Integration Key', $token?->name);
        $this->assertIsArray($token?->abilities);
        $this->assertContains('tasks.read', $token?->abilities ?? []);
        $this->assertContains('tasks.update', $token?->abilities ?? []);
        $this->assertContains('disks.read', $token?->abilities ?? []);

        $this->actingAs($user)
            ->delete(route('profile.api-tokens.destroy', $token))
            ->assertRedirect()
            ->assertSessionHasNoErrors();

        $this->assertDatabaseMissing('personal_access_tokens', [
            'id' => $token?->id,
        ]);
    }
}
