<?php

namespace Tests\Feature;

use App\Models\AccessGroup;
use App\Models\User;
use App\Support\MenuManager;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class MenuPermissionVisibilityTest extends TestCase
{
    use RefreshDatabase;

    public function test_sidebar_hides_sections_without_read_permissions_for_regular_user(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
        ]);

        $keys = app(MenuManager::class)
            ->sidebarItems($user)
            ->pluck('key')
            ->filter()
            ->values()
            ->all();

        $this->assertContains('tasks', $keys);
        $this->assertNotContains('companies', $keys);
        $this->assertNotContains('contacts', $keys);
        $this->assertNotContains('deals', $keys);
        $this->assertNotContains('products', $keys);
        $this->assertNotContains('warehouses', $keys);
        $this->assertNotContains('hr', $keys);
        $this->assertNotContains('projects', $keys);
        $this->assertNotContains('activities', $keys);
        $this->assertNotContains('pipelines', $keys);
        $this->assertNotContains('onec', $keys);
        $this->assertNotContains('mail', $keys);
        $this->assertNotContains('reports', $keys);
        $this->assertNotContains('forms', $keys);
        $this->assertNotContains('disks', $keys);
        $this->assertNotContains('calendar', $keys);
    }

    public function test_sidebar_shows_section_when_personal_read_permission_is_granted(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'deals.read' => true,
            ],
        ]);

        $keys = app(MenuManager::class)
            ->sidebarItems($user)
            ->pluck('key')
            ->filter()
            ->values()
            ->all();

        $this->assertContains('deals', $keys);
    }

    public function test_sidebar_hides_task_section_when_personal_read_permission_is_revoked(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'tasks.read' => false,
            ],
        ]);

        $keys = app(MenuManager::class)
            ->sidebarItems($user)
            ->pluck('key')
            ->filter()
            ->values()
            ->all();

        $this->assertNotContains('tasks', $keys);
    }

    public function test_sidebar_shows_section_when_group_read_permission_is_granted(): void
    {
        $group = AccessGroup::query()->create([
            'name' => 'Deals readers',
            'permissions' => [
                'deals.read' => true,
            ],
        ]);

        $user = User::factory()->create([
            'role' => 'user',
            'access_group_id' => $group->id,
        ]);

        $keys = app(MenuManager::class)
            ->sidebarItems($user)
            ->pluck('key')
            ->filter()
            ->values()
            ->all();

        $this->assertContains('deals', $keys);
    }
}
