<?php

namespace Tests\Feature;

use App\Models\Company;
use App\Models\Deal;
use App\Models\DealStage;
use App\Models\Pipeline;
use App\Models\Task;
use App\Models\User;
use App\Models\WebForm;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class FormWorkbenchTest extends TestCase
{
    use RefreshDatabase;

    public function test_form_create_page_starts_with_single_default_field_and_add_button(): void
    {
        $user = User::factory()->create(['role' => 'manager']);

        $this->actingAs($user)
            ->get(route('forms.create'))
            ->assertOk()
            ->assertSee('data-add-field', false)
            ->assertSee('name="fields[0][label]"', false)
            ->assertDontSee('name="fields[1][label]"', false);
    }

    public function test_user_can_create_form_and_see_public_link(): void
    {
        $user = User::factory()->create(['role' => 'manager']);

        $this->actingAs($user)
            ->post(route('forms.store'), [
                'name' => 'Lead capture',
                'slug' => 'lead-capture',
                'submit_label' => 'Send request',
                'show_reset_button' => 1,
                'reset_label' => 'Clear form',
                'success_message' => 'Thanks!',
                'is_active' => 1,
                'behavior' => 'create_task',
                'behavior_settings' => [
                    'priority' => 'high',
                ],
                'fields' => [
                    [
                        'label' => 'Subject',
                        'name' => 'subject',
                        'type' => 'text',
                        'required' => 1,
                        'placeholder' => 'Subject',
                        'target' => 'title',
                        'options' => '',
                    ],
                ],
            ])
            ->assertRedirect();

        $this->assertDatabaseHas('web_forms', [
            'name' => 'Lead capture',
            'slug' => 'lead-capture',
            'owner_id' => $user->id,
            'behavior' => 'create_task',
        ]);

        $this->actingAs($user)
            ->get(route('forms.index'))
            ->assertOk()
            ->assertSee(route('public-forms.show', ['form' => 'lead-capture']), false);
    }

    public function test_public_form_submission_can_create_company(): void
    {
        $owner = User::factory()->create(['role' => 'manager']);

        $form = WebForm::query()->create([
            'owner_id' => $owner->id,
            'name' => 'Company intake',
            'slug' => 'company-intake',
            'submit_label' => 'Submit',
            'show_reset_button' => false,
            'reset_label' => 'Clear',
            'success_message' => 'Received',
            'is_active' => true,
            'behavior' => 'create_company',
            'behavior_settings' => [
                'owner_id' => $owner->id,
                'status' => 'lead',
            ],
            'fields' => [
                ['label' => 'Company', 'name' => 'company', 'type' => 'text', 'required' => true, 'placeholder' => '', 'target' => 'name', 'options' => []],
                ['label' => 'Email', 'name' => 'email', 'type' => 'email', 'required' => false, 'placeholder' => '', 'target' => 'email', 'options' => []],
                ['label' => 'Phone', 'name' => 'phone', 'type' => 'text', 'required' => false, 'placeholder' => '', 'target' => 'phone', 'options' => []],
            ],
        ]);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'company' => 'Acme Labs',
            'email' => 'hello@acme.test',
            'phone' => '+1-555-3000',
        ])
            ->assertRedirect()
            ->assertSessionHasNoErrors();

        $this->assertDatabaseHas('companies', [
            'name' => 'Acme Labs',
            'email' => 'hello@acme.test',
            'owner_id' => $owner->id,
            'status' => 'lead',
        ]);

        $this->assertDatabaseHas('web_form_submissions', [
            'web_form_id' => $form->id,
            'status' => 'success',
            'result_type' => 'company',
        ]);
    }

    public function test_public_form_field_map_can_fill_task_fields(): void
    {
        $owner = User::factory()->create(['role' => 'manager']);
        $assignee = User::factory()->create(['role' => 'user']);

        $form = WebForm::query()->create([
            'owner_id' => $owner->id,
            'name' => 'Task intake',
            'slug' => 'task-intake',
            'submit_label' => 'Submit',
            'show_reset_button' => false,
            'reset_label' => 'Clear',
            'success_message' => 'Received',
            'is_active' => true,
            'behavior' => 'create_task',
            'behavior_settings' => [
                'assignee_id' => $owner->id,
                'priority' => 'low',
                'field_map' => [
                    ['crm_field' => 'title', 'form_field' => 'topic'],
                    ['crm_field' => 'due_at', 'form_field' => 'deadline'],
                    ['crm_field' => 'priority', 'form_field' => 'priority_code'],
                    ['crm_field' => 'status', 'form_field' => 'status_code'],
                    ['crm_field' => 'assignee_id', 'form_field' => 'assignee_code'],
                ],
            ],
            'fields' => [
                ['label' => 'Topic', 'name' => 'topic', 'type' => 'text', 'required' => true, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Deadline', 'name' => 'deadline', 'type' => 'date', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Priority', 'name' => 'priority_code', 'type' => 'text', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Status', 'name' => 'status_code', 'type' => 'text', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Assignee', 'name' => 'assignee_code', 'type' => 'number', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
            ],
        ]);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'topic' => 'Call customer',
            'deadline' => '2026-03-04',
            'priority_code' => 'high',
            'status_code' => 'in_progress',
            'assignee_code' => (string) $assignee->id,
        ])
            ->assertRedirect()
            ->assertSessionHasNoErrors();

        /** @var Task $task */
        $task = Task::query()->latest('id')->firstOrFail();

        $this->assertSame('Call customer', $task->title);
        $this->assertSame('high', $task->priority);
        $this->assertSame('in_progress', $task->status);
        $this->assertSame($assignee->id, (int) $task->assignee_id);
        $this->assertSame('2026-03-04', optional($task->due_at)->toDateString());
    }

    public function test_public_form_field_map_can_fill_deal_fields(): void
    {
        $owner = User::factory()->create(['role' => 'manager']);
        $pipeline = Pipeline::factory()->create(['is_default' => false]);
        $stage = DealStage::factory()->create([
            'pipeline_id' => $pipeline->id,
            'sort_order' => 1,
        ]);

        $form = WebForm::query()->create([
            'owner_id' => $owner->id,
            'name' => 'Deal intake',
            'slug' => 'deal-intake',
            'submit_label' => 'Submit',
            'show_reset_button' => false,
            'reset_label' => 'Clear',
            'success_message' => 'Received',
            'is_active' => true,
            'behavior' => 'create_deal',
            'behavior_settings' => [
                'owner_id' => $owner->id,
                'currency' => 'USD',
                'field_map' => [
                    ['crm_field' => 'title', 'form_field' => 'lead_title'],
                    ['crm_field' => 'amount', 'form_field' => 'budget'],
                    ['crm_field' => 'currency', 'form_field' => 'currency_code'],
                    ['crm_field' => 'pipeline_id', 'form_field' => 'pipeline_code'],
                    ['crm_field' => 'stage_id', 'form_field' => 'stage_code'],
                    ['crm_field' => 'expected_close_at', 'form_field' => 'close_date'],
                    ['crm_field' => 'priority', 'form_field' => 'priority_code'],
                    ['crm_field' => 'status', 'form_field' => 'status_code'],
                ],
            ],
            'fields' => [
                ['label' => 'Title', 'name' => 'lead_title', 'type' => 'text', 'required' => true, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Budget', 'name' => 'budget', 'type' => 'number', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Currency', 'name' => 'currency_code', 'type' => 'text', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Pipeline', 'name' => 'pipeline_code', 'type' => 'number', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Stage', 'name' => 'stage_code', 'type' => 'number', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Close date', 'name' => 'close_date', 'type' => 'date', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Priority', 'name' => 'priority_code', 'type' => 'text', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Status', 'name' => 'status_code', 'type' => 'text', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
            ],
        ]);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'lead_title' => 'Enterprise deal',
            'budget' => '4200',
            'currency_code' => 'eur',
            'pipeline_code' => (string) $pipeline->id,
            'stage_code' => (string) $stage->id,
            'close_date' => '2026-04-22',
            'priority_code' => 'high',
            'status_code' => 'open',
        ])
            ->assertRedirect()
            ->assertSessionHasNoErrors();

        /** @var Deal $deal */
        $deal = Deal::query()->latest('id')->firstOrFail();

        $this->assertSame('Enterprise deal', $deal->title);
        $this->assertSame($pipeline->id, (int) $deal->pipeline_id);
        $this->assertSame($stage->id, (int) $deal->stage_id);
        $this->assertSame('EUR', $deal->currency);
        $this->assertSame('high', $deal->priority);
        $this->assertSame('open', $deal->status);
        $this->assertSame('2026-04-22', optional($deal->expected_close_at)->toDateString());
    }

    public function test_public_form_field_map_can_override_company_status_and_name(): void
    {
        $owner = User::factory()->create(['role' => 'manager']);

        $form = WebForm::query()->create([
            'owner_id' => $owner->id,
            'name' => 'Company advanced intake',
            'slug' => 'company-advanced-intake',
            'submit_label' => 'Submit',
            'show_reset_button' => false,
            'reset_label' => 'Clear',
            'success_message' => 'Received',
            'is_active' => true,
            'behavior' => 'create_company',
            'behavior_settings' => [
                'owner_id' => $owner->id,
                'status' => 'lead',
                'field_map' => [
                    ['crm_field' => 'name', 'form_field' => 'client_name'],
                    ['crm_field' => 'email', 'form_field' => 'client_email'],
                    ['crm_field' => 'status', 'form_field' => 'client_status'],
                ],
            ],
            'fields' => [
                ['label' => 'Client name', 'name' => 'client_name', 'type' => 'text', 'required' => true, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Client email', 'name' => 'client_email', 'type' => 'email', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
                ['label' => 'Client status', 'name' => 'client_status', 'type' => 'text', 'required' => false, 'placeholder' => '', 'target' => '', 'options' => []],
            ],
        ]);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'client_name' => 'Northwind',
            'client_email' => 'sales@northwind.test',
            'client_status' => 'client',
        ])
            ->assertRedirect()
            ->assertSessionHasNoErrors();

        /** @var Company $company */
        $company = Company::query()->latest('id')->firstOrFail();

        $this->assertSame('Northwind', $company->name);
        $this->assertSame('sales@northwind.test', $company->email);
        $this->assertSame('client', $company->status);
    }

    public function test_form_store_persists_submit_result_settings(): void
    {
        $user = User::factory()->create(['role' => 'manager']);

        $this->actingAs($user)
            ->post(route('forms.store'), [
                'name' => 'Follow-up form',
                'slug' => 'follow-up-form',
                'submit_label' => 'Submit',
                'show_reset_button' => 0,
                'reset_label' => 'Clear',
                'success_message' => 'Saved',
                'is_active' => 1,
                'behavior' => 'create_task',
                'behavior_settings' => [
                    'submit_result_action' => 'redirect',
                    'redirect_url' => '/thank-you',
                    'priority' => 'high',
                ],
                'fields' => [
                    [
                        'label' => 'Topic',
                        'name' => 'topic',
                        'type' => 'text',
                        'required' => 1,
                        'placeholder' => '',
                        'target' => 'title',
                        'options' => '',
                    ],
                ],
            ])
            ->assertRedirect();

        /** @var WebForm $form */
        $form = WebForm::query()->where('slug', 'follow-up-form')->firstOrFail();
        $settings = is_array($form->behavior_settings) ? $form->behavior_settings : [];

        $this->assertSame('redirect', $settings['submit_result_action'] ?? null);
        $this->assertSame('/thank-you', $settings['redirect_url'] ?? null);
        $this->assertSame('high', $settings['priority'] ?? null);
    }

    public function test_public_form_hide_form_action_locks_repeat_submission_in_same_session(): void
    {
        $owner = User::factory()->create(['role' => 'manager']);

        $form = WebForm::query()->create([
            'owner_id' => $owner->id,
            'name' => 'Single submit form',
            'slug' => 'single-submit-form',
            'submit_label' => 'Submit',
            'show_reset_button' => false,
            'reset_label' => 'Clear',
            'success_message' => 'Received',
            'is_active' => true,
            'behavior' => 'create_task',
            'behavior_settings' => [
                'submit_result_action' => 'hide_form',
            ],
            'fields' => [
                ['label' => 'Subject', 'name' => 'subject', 'type' => 'text', 'required' => true, 'placeholder' => '', 'target' => 'title', 'options' => []],
            ],
        ]);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'subject' => 'One-time request',
        ])
            ->assertRedirect(route('public-forms.show', ['form' => $form->slug]))
            ->assertSessionHasNoErrors();

        $this->assertDatabaseCount('web_form_submissions', 1);
        $this->assertDatabaseCount('tasks', 1);

        $this->get(route('public-forms.show', ['form' => $form->slug]))
            ->assertOk()
            ->assertSee('This form has already been submitted from this browser session.')
            ->assertDontSee('name="subject"', false);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'subject' => 'Repeat request',
        ])->assertRedirect(route('public-forms.show', ['form' => $form->slug]));

        $this->assertDatabaseCount('web_form_submissions', 1);
        $this->assertDatabaseCount('tasks', 1);
    }

    public function test_public_form_redirect_action_redirects_to_configured_url(): void
    {
        $owner = User::factory()->create(['role' => 'manager']);

        $form = WebForm::query()->create([
            'owner_id' => $owner->id,
            'name' => 'Redirect form',
            'slug' => 'redirect-form',
            'submit_label' => 'Submit',
            'show_reset_button' => false,
            'reset_label' => 'Clear',
            'success_message' => 'Received',
            'is_active' => true,
            'behavior' => 'create_task',
            'behavior_settings' => [
                'submit_result_action' => 'redirect',
                'redirect_url' => '/after-submit',
            ],
            'fields' => [
                ['label' => 'Subject', 'name' => 'subject', 'type' => 'text', 'required' => true, 'placeholder' => '', 'target' => 'title', 'options' => []],
            ],
        ]);

        $this->post(route('public-forms.submit', ['form' => $form->slug]), [
            'subject' => 'Redirect me',
        ])
            ->assertRedirect('/after-submit')
            ->assertSessionHasNoErrors();

        $this->assertDatabaseHas('web_form_submissions', [
            'web_form_id' => $form->id,
            'status' => 'success',
            'result_type' => 'task',
        ]);
    }
}
