<?php

namespace Tests\Feature;

use App\Models\Disk;
use App\Models\DiskFolder;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;
use ZipArchive;

class DiskFeatureTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_without_disk_read_permission_cannot_open_disk_page(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
        ]);

        $this->actingAs($user)
            ->get(route('disks.index'))
            ->assertForbidden();
    }

    public function test_user_with_personal_disk_permissions_can_upload_and_delete_file(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
                'disks.create' => true,
                'disks.update' => true,
                'disks.delete' => true,
            ],
        ]);

        $file = UploadedFile::fake()->create('contract.pdf', 120, 'application/pdf');

        $this->actingAs($user)
            ->post(route('disks.store'), [
                'file' => $file,
                'name' => 'Contract #1',
                'folder' => 'shared/contracts',
                'description' => 'Signed document',
                'is_public' => 0,
            ])
            ->assertRedirect(route('disks.index'));

        $disk = Disk::query()->where('name', 'Contract #1')->first();

        $this->assertNotNull($disk);
        $this->assertSame($user->id, $disk?->owner_id);
        $this->assertSame('shared/contracts', $disk?->folder);
        Storage::disk('public')->assertExists((string) $disk?->storage_path);

        $this->actingAs($user)
            ->delete(route('disks.destroy', $disk))
            ->assertRedirect(route('disks.index'));

        $this->assertDatabaseMissing('disks', [
            'id' => $disk?->id,
        ]);
        Storage::disk('public')->assertMissing((string) $disk?->storage_path);
    }

    public function test_disk_upload_form_contains_drag_and_drop_area(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
                'disks.create' => true,
            ],
        ]);

        $this->actingAs($user)
            ->get(route('disks.index'))
            ->assertOk()
            ->assertSee('data-disk-dropzone', false)
            ->assertSee('data-disk-file-input', false)
            ->assertSee('data-disk-drop-icon', false)
            ->assertSee('Drop file in the center area');
    }

    public function test_disk_list_contains_copy_link_action_for_file(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $storedPath = UploadedFile::fake()->create('copy-me.pdf', 24, 'application/pdf')
            ->store('disk/shared', 'public');

        $disk = Disk::query()->create([
            'name' => 'Copy me',
            'original_name' => 'copy-me.pdf',
            'storage_path' => $storedPath,
            'folder' => 'shared',
            'mime_type' => 'application/pdf',
            'extension' => 'pdf',
            'size' => 24576,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->get(route('disks.index'))
            ->assertOk()
            ->assertSee('data-disk-copy-link', false)
            ->assertSee(route('disks.preview', $disk), false);
    }

    public function test_disk_page_renders_explorer_tree_and_context_menu(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $this->actingAs($user)
            ->get(route('disks.index'))
            ->assertOk()
            ->assertSee('data-disk-explorer', false)
            ->assertSee('data-folder-node', false)
            ->assertSee('data-explorer-context-menu', false);
    }

    public function test_user_can_move_file_with_explorer_endpoint(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
                'disks.update' => true,
                'disks.create' => true,
            ],
        ]);

        $path = UploadedFile::fake()->create('invoice.pdf', 32, 'application/pdf')
            ->store('disk/contracts', 'public');

        $disk = Disk::query()->create([
            'name' => 'Invoice',
            'original_name' => 'invoice.pdf',
            'storage_path' => $path,
            'folder' => 'contracts',
            'mime_type' => 'application/pdf',
            'extension' => 'pdf',
            'size' => 32768,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->patchJson(route('disks.explorer.files.move', $disk), [
                'destination_folder' => 'sales/archive',
            ])
            ->assertOk()
            ->assertJsonPath('file.folder', 'sales/archive');

        $disk = $disk->fresh();
        $this->assertSame('sales/archive', $disk?->folder);
        Storage::disk('public')->assertExists((string) $disk?->storage_path);
        $this->assertDatabaseHas('disk_folders', [
            'owner_id' => $user->id,
            'path' => 'sales/archive',
        ]);
    }

    public function test_user_can_copy_and_delete_folder_with_explorer_endpoints(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
                'disks.create' => true,
                'disks.update' => true,
                'disks.delete' => true,
            ],
        ]);

        DiskFolder::query()->create([
            'name' => 'contracts',
            'path' => 'contracts',
            'parent_path' => null,
            'owner_id' => $user->id,
        ]);

        $path = UploadedFile::fake()->create('contract.pdf', 20, 'application/pdf')
            ->store('disk/contracts', 'public');

        Disk::query()->create([
            'name' => 'Contract',
            'original_name' => 'contract.pdf',
            'storage_path' => $path,
            'folder' => 'contracts',
            'mime_type' => 'application/pdf',
            'extension' => 'pdf',
            'size' => 20480,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $copyResponse = $this->actingAs($user)
            ->postJson(route('disks.explorer.folders.copy'), [
                'source_folder' => 'contracts',
                'destination_folder' => '',
            ])
            ->assertOk();

        $copiedFolderPath = (string) $copyResponse->json('folder.path');
        $this->assertNotSame('', $copiedFolderPath);
        $this->assertDatabaseHas('disk_folders', [
            'owner_id' => $user->id,
            'path' => $copiedFolderPath,
        ]);
        $this->assertSame(2, Disk::query()->where('owner_id', $user->id)->count());

        $this->actingAs($user)
            ->deleteJson(route('disks.explorer.folders.delete'), [
                'folder' => 'contracts',
            ])
            ->assertOk();

        $this->assertDatabaseMissing('disk_folders', [
            'owner_id' => $user->id,
            'path' => 'contracts',
        ]);
        $this->assertDatabaseHas('disk_folders', [
            'owner_id' => $user->id,
            'path' => $copiedFolderPath,
        ]);
        $this->assertSame(1, Disk::query()->where('owner_id', $user->id)->count());
    }

    public function test_disk_list_renders_file_type_icon_for_uploaded_file(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $storedPath = UploadedFile::fake()->create('invoice.pdf', 16, 'application/pdf')
            ->store('disk/shared', 'public');

        Disk::query()->create([
            'name' => 'Invoice',
            'original_name' => 'invoice.pdf',
            'storage_path' => $storedPath,
            'folder' => 'shared',
            'mime_type' => 'application/pdf',
            'extension' => 'pdf',
            'size' => 16384,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->get(route('disks.index'))
            ->assertOk()
            ->assertSee('data-disk-file-icon="fa-solid fa-file-pdf"', false);
    }

    public function test_user_can_create_disk_catalog_and_see_it_in_filter_list(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
                'disks.create' => true,
            ],
        ]);

        $this->actingAs($user)
            ->post(route('disks.folders.store'), [
                'catalog_name' => 'Contracts',
                'parent_folder' => 'sales',
            ])
            ->assertRedirect(route('disks.index', ['folder' => 'sales/Contracts']));

        $folder = DiskFolder::query()
            ->where('owner_id', $user->id)
            ->where('path', 'sales/Contracts')
            ->first();

        $this->assertNotNull($folder);
        $this->assertSame('Contracts', $folder?->name);
        $this->assertSame('sales', $folder?->parent_path);
        Storage::disk('public')->assertExists('disk/sales/Contracts');

        $this->actingAs($user)
            ->get(route('disks.index'))
            ->assertOk()
            ->assertSee('sales/Contracts');
    }

    public function test_user_can_open_pdf_preview_page_and_inline_stream(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $storedPath = UploadedFile::fake()->create('presentation.pdf', 32, 'application/pdf')
            ->store('disk/shared', 'public');

        $disk = Disk::query()->create([
            'name' => 'Presentation',
            'original_name' => 'presentation.pdf',
            'storage_path' => $storedPath,
            'folder' => 'shared',
            'mime_type' => 'application/pdf',
            'extension' => 'pdf',
            'size' => 32768,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->get(route('disks.preview', $disk))
            ->assertOk()
            ->assertSee('PDF preview')
            ->assertSee(route('disks.inline', $disk), false);

        $this->actingAs($user)
            ->get(route('disks.inline', $disk))
            ->assertOk()
            ->assertHeader('content-type', 'application/pdf');
    }

    public function test_user_can_preview_docx_inside_crm_page(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $disk = $this->createDocxDisk($user, 'Commercial Proposal');

        $this->actingAs($user)
            ->get(route('disks.preview', $disk))
            ->assertOk()
            ->assertSee('Word preview')
            ->assertSee('data-disk-word-preview', false)
            ->assertSee(route('disks.word-source', $disk), false);

        $this->actingAs($user)
            ->get(route('disks.word-source', $disk))
            ->assertOk()
            ->assertHeader('content-type', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');
    }

    public function test_user_can_preview_doc_inside_crm_page_when_converter_is_available(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $fixtureDocxPath = Storage::disk('public')->path('disk/shared/doc-converted.docx');
        @mkdir(dirname($fixtureDocxPath), 0777, true);
        $this->createDocxFile($fixtureDocxPath, 'Converted DOC content');

        $converterPath = $this->createFakeDocConverterScript($fixtureDocxPath);
        putenv('DOC_PREVIEW_CONVERTER_PATH='.$converterPath);
        $_ENV['DOC_PREVIEW_CONVERTER_PATH'] = $converterPath;
        $_SERVER['DOC_PREVIEW_CONVERTER_PATH'] = $converterPath;
        try {
            $disk = $this->createDocDisk($user, 'preview.doc');

            $this->actingAs($user)
                ->get(route('disks.preview', $disk))
                ->assertOk()
                ->assertSee('Word preview')
                ->assertSee('data-disk-word-preview', false)
                ->assertSee(route('disks.word-source', $disk), false);

            $this->actingAs($user)
                ->get(route('disks.word-source', $disk))
                ->assertOk()
                ->assertHeader('content-type', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');
        } finally {
            putenv('DOC_PREVIEW_CONVERTER_PATH');
            unset($_ENV['DOC_PREVIEW_CONVERTER_PATH'], $_SERVER['DOC_PREVIEW_CONVERTER_PATH']);
        }
    }

    public function test_user_can_preview_xlsx_inside_crm_page(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $disk = $this->createXlsxDisk($user, 'Revenue');

        $this->actingAs($user)
            ->get(route('disks.preview', $disk))
            ->assertOk()
            ->assertSee('Spreadsheet preview')
            ->assertSee('data-disk-spreadsheet-preview', false)
            ->assertSee(route('disks.spreadsheet-source', $disk), false);

        $this->actingAs($user)
            ->get(route('disks.spreadsheet-source', $disk))
            ->assertOk()
            ->assertHeader('content-type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    }

    public function test_user_can_preview_csv_inside_crm_page_with_spreadsheet_viewer(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $path = 'disk/shared/preview.csv';
        $absolutePath = Storage::disk('public')->path($path);
        @mkdir(dirname($absolutePath), 0777, true);
        file_put_contents($absolutePath, "name,amount\nRevenue,125000\n");

        $disk = Disk::query()->create([
            'name' => 'preview',
            'original_name' => 'preview.csv',
            'storage_path' => $path,
            'folder' => 'shared',
            'mime_type' => 'text/csv',
            'extension' => 'csv',
            'size' => filesize($absolutePath) ?: 0,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->get(route('disks.preview', $disk))
            ->assertOk()
            ->assertSee('Spreadsheet preview')
            ->assertSee('data-disk-spreadsheet-preview', false)
            ->assertSee('data-extension="csv"', false)
            ->assertSee(route('disks.spreadsheet-source', $disk), false);

        $this->actingAs($user)
            ->get(route('disks.spreadsheet-source', $disk))
            ->assertOk()
            ->assertHeader('content-type', 'text/csv; charset=UTF-8');
    }

    public function test_user_can_preview_xls_inside_crm_page_when_converter_is_available(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $fixtureXlsxPath = Storage::disk('public')->path('disk/shared/xls-converted.xlsx');
        @mkdir(dirname($fixtureXlsxPath), 0777, true);
        $this->createXlsxFile($fixtureXlsxPath, 'Converted XLS');

        $converterPath = $this->createFakeXlsConverterScript($fixtureXlsxPath);
        putenv('OFFICE_PREVIEW_CONVERTER_PATH='.$converterPath);
        $_ENV['OFFICE_PREVIEW_CONVERTER_PATH'] = $converterPath;
        $_SERVER['OFFICE_PREVIEW_CONVERTER_PATH'] = $converterPath;
        try {
            $path = 'disk/shared/preview.xls';
            $absolutePath = Storage::disk('public')->path($path);
            @mkdir(dirname($absolutePath), 0777, true);
            file_put_contents($absolutePath, "xls-binary-placeholder\n");

            $disk = Disk::query()->create([
                'name' => 'preview',
                'original_name' => 'preview.xls',
                'storage_path' => $path,
                'folder' => 'shared',
                'mime_type' => 'application/vnd.ms-excel',
                'extension' => 'xls',
                'size' => filesize($absolutePath) ?: 0,
                'description' => null,
                'is_public' => false,
                'owner_id' => $user->id,
            ]);

            $this->actingAs($user)
                ->get(route('disks.preview', $disk))
                ->assertOk()
                ->assertSee('Spreadsheet preview')
                ->assertSee('data-disk-spreadsheet-preview', false)
                ->assertSee(route('disks.spreadsheet-source', $disk), false);

            $this->actingAs($user)
                ->get(route('disks.spreadsheet-source', $disk))
                ->assertOk()
                ->assertHeader('content-type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        } finally {
            putenv('OFFICE_PREVIEW_CONVERTER_PATH');
            unset($_ENV['OFFICE_PREVIEW_CONVERTER_PATH'], $_SERVER['OFFICE_PREVIEW_CONVERTER_PATH']);
        }
    }

    public function test_xls_preview_returns_422_when_converter_is_not_available(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'disks.read' => true,
            ],
        ]);

        $path = 'disk/shared/preview.xls';
        $absolutePath = Storage::disk('public')->path($path);
        @mkdir(dirname($absolutePath), 0777, true);
        file_put_contents($absolutePath, "xls-binary-placeholder\n");

        $disk = Disk::query()->create([
            'name' => 'preview',
            'original_name' => 'preview.xls',
            'storage_path' => $path,
            'folder' => 'shared',
            'mime_type' => 'application/vnd.ms-excel',
            'extension' => 'xls',
            'size' => filesize($absolutePath) ?: 0,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $originalPath = getenv('PATH') ?: '';
        $originalOfficeConverter = getenv('OFFICE_PREVIEW_CONVERTER_PATH');
        $originalDocConverter = getenv('DOC_PREVIEW_CONVERTER_PATH');

        putenv('PATH=');
        putenv('OFFICE_PREVIEW_CONVERTER_PATH=');
        putenv('DOC_PREVIEW_CONVERTER_PATH=');
        $_ENV['OFFICE_PREVIEW_CONVERTER_PATH'] = '';
        $_SERVER['OFFICE_PREVIEW_CONVERTER_PATH'] = '';
        $_ENV['DOC_PREVIEW_CONVERTER_PATH'] = '';
        $_SERVER['DOC_PREVIEW_CONVERTER_PATH'] = '';

        try {
            $this->actingAs($user)
                ->get(route('disks.spreadsheet-source', $disk))
                ->assertStatus(422)
                ->assertSeeText('XLS preview requires LibreOffice (soffice) on server.');
        } finally {
            putenv('PATH='.$originalPath);

            if ($originalOfficeConverter === false) {
                putenv('OFFICE_PREVIEW_CONVERTER_PATH');
                unset($_ENV['OFFICE_PREVIEW_CONVERTER_PATH'], $_SERVER['OFFICE_PREVIEW_CONVERTER_PATH']);
            } else {
                putenv('OFFICE_PREVIEW_CONVERTER_PATH='.$originalOfficeConverter);
                $_ENV['OFFICE_PREVIEW_CONVERTER_PATH'] = $originalOfficeConverter;
                $_SERVER['OFFICE_PREVIEW_CONVERTER_PATH'] = $originalOfficeConverter;
            }

            if ($originalDocConverter === false) {
                putenv('DOC_PREVIEW_CONVERTER_PATH');
                unset($_ENV['DOC_PREVIEW_CONVERTER_PATH'], $_SERVER['DOC_PREVIEW_CONVERTER_PATH']);
            } else {
                putenv('DOC_PREVIEW_CONVERTER_PATH='.$originalDocConverter);
                $_ENV['DOC_PREVIEW_CONVERTER_PATH'] = $originalDocConverter;
                $_SERVER['DOC_PREVIEW_CONVERTER_PATH'] = $originalDocConverter;
            }
        }
    }

    private function createDocDisk(User $user, string $originalName): Disk
    {
        $path = 'disk/shared/'.$originalName;
        $absolutePath = Storage::disk('public')->path($path);
        @mkdir(dirname($absolutePath), 0777, true);
        file_put_contents($absolutePath, "Fake legacy DOC payload\n");

        return Disk::query()->create([
            'name' => 'preview',
            'original_name' => $originalName,
            'storage_path' => $path,
            'folder' => 'shared',
            'mime_type' => 'application/msword',
            'extension' => 'doc',
            'size' => filesize($absolutePath) ?: 0,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);
    }

    private function createDocxDisk(User $user, string $text): Disk
    {
        $path = 'disk/shared/preview.docx';
        $absolutePath = Storage::disk('public')->path($path);
        @mkdir(dirname($absolutePath), 0777, true);

        $this->createDocxFile($absolutePath, $text);

        return Disk::query()->create([
            'name' => 'preview',
            'original_name' => 'preview.docx',
            'storage_path' => $path,
            'folder' => 'shared',
            'mime_type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'extension' => 'docx',
            'size' => filesize($absolutePath) ?: 0,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);
    }

    private function createDocxFile(string $absolutePath, string $text): void
    {
        $zip = new ZipArchive();
        $this->assertTrue($zip->open($absolutePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true);
        $zip->addFromString('[Content_Types].xml', <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
  <Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>
  <Default Extension="xml" ContentType="application/xml"/>
  <Override PartName="/word/document.xml" ContentType="application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml"/>
</Types>
XML);
        $zip->addFromString('_rels/.rels', <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
  <Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="word/document.xml"/>
</Relationships>
XML);
        $zip->addFromString('word/document.xml', '<?xml version="1.0" encoding="UTF-8"?><w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main"><w:body><w:p><w:r><w:t>'.htmlspecialchars($text, ENT_XML1).'</w:t></w:r></w:p></w:body></w:document>');
        $zip->close();
    }

    private function createXlsxFile(string $absolutePath, string $label): void
    {
        $zip = new ZipArchive();
        $this->assertTrue($zip->open($absolutePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true);
        $zip->addFromString('[Content_Types].xml', <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
  <Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>
  <Default Extension="xml" ContentType="application/xml"/>
  <Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>
  <Override PartName="/xl/worksheets/sheet1.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>
  <Override PartName="/xl/sharedStrings.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml"/>
</Types>
XML);
        $zip->addFromString('_rels/.rels', <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
  <Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="xl/workbook.xml"/>
</Relationships>
XML);
        $zip->addFromString('xl/workbook.xml', <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">
  <sheets>
    <sheet name="Sheet1" sheetId="1" r:id="rId1"/>
  </sheets>
</workbook>
XML);
        $zip->addFromString('xl/_rels/workbook.xml.rels', <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
  <Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet" Target="worksheets/sheet1.xml"/>
</Relationships>
XML);
        $zip->addFromString('xl/sharedStrings.xml', '<?xml version="1.0" encoding="UTF-8"?><sst xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" count="1" uniqueCount="1"><si><t>'.htmlspecialchars($label, ENT_XML1).'</t></si></sst>');
        $zip->addFromString('xl/worksheets/sheet1.xml', <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">
  <sheetData>
    <row r="1">
      <c r="A1" t="s"><v>0</v></c>
      <c r="B1"><v>125000</v></c>
    </row>
  </sheetData>
</worksheet>
XML);
        $zip->close();
    }

    private function createFakeDocConverterScript(string $fixtureDocxPath): string
    {
        $scriptsDirectory = Storage::disk('public')->path('disk/testing');
        @mkdir($scriptsDirectory, 0777, true);

        $scriptPath = $scriptsDirectory.'/fake-soffice.sh';
        $script = <<<'SH'
#!/bin/sh
set -e
outdir=""
input=""
while [ "$#" -gt 0 ]; do
  case "$1" in
    --outdir)
      outdir="$2"
      shift 2
      ;;
    --convert-to)
      shift 2
      ;;
    --headless)
      shift
      ;;
    --*)
      shift
      ;;
    *)
      input="$1"
      shift
      ;;
  esac
done
if [ -z "$outdir" ] || [ -z "$input" ]; then
  exit 1
fi
name="$(basename "$input")"
name="${name%.*}"
cp __FIXTURE__ "$outdir/$name.docx"
SH;
        $script = str_replace('__FIXTURE__', escapeshellarg($fixtureDocxPath), $script);

        file_put_contents($scriptPath, $script);
        @chmod($scriptPath, 0755);

        return $scriptPath;
    }

    private function createFakeXlsConverterScript(string $fixtureXlsxPath): string
    {
        $scriptsDirectory = Storage::disk('public')->path('disk/testing');
        @mkdir($scriptsDirectory, 0777, true);

        $scriptPath = $scriptsDirectory.'/fake-soffice-xls.sh';
        $script = <<<'SH'
#!/bin/sh
set -e
outdir=""
input=""
while [ "$#" -gt 0 ]; do
  case "$1" in
    --outdir)
      outdir="$2"
      shift 2
      ;;
    --convert-to)
      shift 2
      ;;
    --headless)
      shift
      ;;
    --*)
      shift
      ;;
    *)
      input="$1"
      shift
      ;;
  esac
done
if [ -z "$outdir" ] || [ -z "$input" ]; then
  exit 1
fi
name="$(basename "$input")"
name="${name%.*}"
cp __FIXTURE__ "$outdir/$name.xlsx"
SH;
        $script = str_replace('__FIXTURE__', escapeshellarg($fixtureXlsxPath), $script);

        file_put_contents($scriptPath, $script);
        @chmod($scriptPath, 0755);

        return $scriptPath;
    }

    private function createXlsxDisk(User $user, string $label): Disk
    {
        $path = 'disk/shared/preview.xlsx';
        $absolutePath = Storage::disk('public')->path($path);
        @mkdir(dirname($absolutePath), 0777, true);
        $this->createXlsxFile($absolutePath, $label);

        return Disk::query()->create([
            'name' => 'preview',
            'original_name' => 'preview.xlsx',
            'storage_path' => $path,
            'folder' => 'shared',
            'mime_type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'extension' => 'xlsx',
            'size' => filesize($absolutePath) ?: 0,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);
    }
}
