<?php

namespace Tests\Feature;

use App\Models\Deal;
use App\Models\Pipeline;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DealWorkbenchTest extends TestCase
{
    use RefreshDatabase;

    public function test_deal_pages_support_list_and_kanban_views(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::factory()->create([
            'name' => 'Main funnel',
            'is_default' => true,
            'sort_order' => 1,
        ]);

        $stage = $pipeline->stages()->create([
            'name' => 'New lead',
            'code' => 'new_lead',
            'sort_order' => 1,
            'probability' => 20,
            'is_won' => false,
            'is_lost' => false,
        ]);

        Deal::factory()->create([
            'title' => 'Transaction for Kanban',
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stage->id,
            'owner_id' => $user->id,
            'status' => 'open',
        ]);

        $this->actingAs($user)
            ->get(route('deals.index', ['view' => 'kanban', 'pipeline_id' => $pipeline->id]))
            ->assertOk()
            ->assertSee('id="deal-board"', false)
            ->assertSee('data-stage-color-picker', false)
            ->assertSee('data-stage-move-handle', false)
            ->assertDontSee('Company / Contact');

        $this->actingAs($user)
            ->get(route('deals.index', ['view' => 'list', 'pipeline_id' => $pipeline->id]))
            ->assertOk()
            ->assertSee('Company / Contact')
            ->assertDontSee('id="deal-board"', false);
    }

    public function test_deal_view_selection_is_saved_and_used_as_default(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::factory()->create([
            'name' => 'Preferred funnel',
            'is_default' => true,
            'sort_order' => 1,
        ]);

        $stage = $pipeline->stages()->create([
            'name' => 'New lead',
            'code' => 'preferred_new_lead',
            'sort_order' => 1,
            'probability' => 20,
            'is_won' => false,
            'is_lost' => false,
        ]);

        Deal::factory()->create([
            'title' => 'Preferred view deal',
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stage->id,
            'owner_id' => $user->id,
            'status' => 'open',
        ]);

        $this->actingAs($user)
            ->get(route('deals.index', ['view' => 'kanban', 'pipeline_id' => $pipeline->id]))
            ->assertOk()
            ->assertSee('id="deal-board"', false);

        $this->assertDatabaseHas('users', [
            'id' => $user->id,
            'preferred_deal_view' => 'kanban',
        ]);

        $this->actingAs($user)
            ->get(route('deals.index', ['pipeline_id' => $pipeline->id]))
            ->assertOk()
            ->assertSee('id="deal-board"', false);
    }

    public function test_invalid_deals_view_falls_back_to_list_mode(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::factory()->create([
            'name' => 'Funnel fallback',
            'is_default' => true,
            'sort_order' => 1,
        ]);

        $pipeline->stages()->create([
            'name' => 'New lead',
            'code' => 'fallback_new_lead',
            'sort_order' => 1,
            'probability' => 20,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $this->actingAs($user)
            ->get(route('deals.index', ['view' => 'unknown', 'pipeline_id' => $pipeline->id]))
            ->assertOk()
            ->assertSee('Company / Contact')
            ->assertDontSee('id="deal-board"', false);
    }

    public function test_pipeline_filter_auto_submits_without_apply_button(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::factory()->create([
            'name' => 'Auto-switch funnel',
            'is_default' => true,
            'sort_order' => 1,
        ]);

        $pipeline->stages()->create([
            'name' => 'New lead',
            'code' => 'auto_switch_new_lead',
            'sort_order' => 1,
            'probability' => 20,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $response = $this->actingAs($user)
            ->get(route('deals.index', ['view' => 'list', 'pipeline_id' => $pipeline->id]))
            ->assertOk();

        $html = $response->getContent();

        $this->assertStringContainsString('onchange="this.form.requestSubmit()"', $html);
        $this->assertStringNotContainsString('>Apply<', $html);
    }
}
