<?php

namespace Tests\Feature;

use App\Models\Deal;
use App\Models\Pipeline;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DealStageColorTest extends TestCase
{
    use RefreshDatabase;

    public function test_manager_can_update_pipeline_stage_colors(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::factory()->create([
            'name' => 'Funnel with flowers',
            'is_default' => true,
        ]);

        $stageLead = $pipeline->stages()->create([
            'name' => 'New lead',
            'code' => 'new_lead_1',
            'sort_order' => 0,
            'probability' => 20,
            'color' => '#3B82F6',
            'is_won' => false,
            'is_lost' => false,
        ]);

        $pipeline->stages()->create([
            'name' => 'Success',
            'code' => 'won_2',
            'sort_order' => 1,
            'probability' => 100,
            'color' => '#10B981',
            'is_won' => true,
            'is_lost' => false,
        ]);

        $pipeline->stages()->create([
            'name' => 'Failure',
            'code' => 'lost_3',
            'sort_order' => 2,
            'probability' => 0,
            'color' => '#EF4444',
            'is_won' => false,
            'is_lost' => true,
        ]);

        $response = $this->actingAs($user)
            ->put(route('pipelines.update', $pipeline), [
                'name' => 'Funnel with flowers',
                'description' => 'Stages color update test',
                'is_active' => 1,
                'is_default' => 1,
                'sort_order' => 0,
                'stages' => "New lead\nSuccess\nFailure",
                'stage_colors' => [
                    $stageLead->id => '#123ABC',
                ],
            ]);

        $response->assertRedirect(route('pipelines.edit', $pipeline));
        $response->assertSessionHasNoErrors();

        $this->assertDatabaseHas('deal_stages', [
            'id' => $stageLead->id,
            'color' => '#123ABC',
        ]);
    }

    public function test_deals_kanban_renders_stage_color(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::factory()->create([
            'name' => 'Kanban colors',
            'is_default' => true,
        ]);

        $stage = $pipeline->stages()->create([
            'name' => 'New lead',
            'code' => 'new_lead_1',
            'sort_order' => 0,
            'probability' => 20,
            'color' => '#A1B2C3',
            'is_won' => false,
            'is_lost' => false,
        ]);

        $pipeline->stages()->create([
            'name' => 'Success',
            'code' => 'won_2',
            'sort_order' => 1,
            'probability' => 100,
            'color' => '#10B981',
            'is_won' => true,
            'is_lost' => false,
        ]);

        $pipeline->stages()->create([
            'name' => 'Failure',
            'code' => 'lost_3',
            'sort_order' => 2,
            'probability' => 0,
            'color' => '#EF4444',
            'is_won' => false,
            'is_lost' => true,
        ]);

        Deal::factory()->create([
            'title' => 'Deal with the color stage',
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stage->id,
            'owner_id' => $user->id,
            'status' => 'open',
        ]);

        $this->actingAs($user)
            ->get(route('deals.index', ['view' => 'kanban', 'pipeline_id' => $pipeline->id]))
            ->assertOk()
            ->assertSee('border-top-color: #A1B2C3;', false);
    }

    public function test_manager_can_update_stage_order_and_colors_from_kanban_endpoint(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::factory()->create([
            'name' => 'Kanban order and color',
            'is_default' => true,
        ]);

        $firstStage = $pipeline->stages()->create([
            'name' => 'New lead',
            'code' => 'new_lead_1',
            'sort_order' => 0,
            'probability' => 20,
            'color' => '#3B82F6',
            'is_won' => false,
            'is_lost' => false,
        ]);

        $secondStage = $pipeline->stages()->create([
            'name' => 'Negotiation',
            'code' => 'negotiation_2',
            'sort_order' => 1,
            'probability' => 60,
            'color' => '#F59E0B',
            'is_won' => false,
            'is_lost' => false,
        ]);

        $wonStage = $pipeline->stages()->create([
            'name' => 'Success',
            'code' => 'won_3',
            'sort_order' => 2,
            'probability' => 100,
            'color' => '#10B981',
            'is_won' => true,
            'is_lost' => false,
        ]);

        $this->actingAs($user)
            ->patchJson(route('pipelines.update-stages', $pipeline), [
                'stages' => [
                    ['id' => $secondStage->id, 'color' => '#123ABC'],
                    ['id' => $firstStage->id, 'color' => '#654321'],
                    ['id' => $wonStage->id, 'color' => '#10B981'],
                ],
            ])
            ->assertOk()
            ->assertJsonPath('stages.0.id', $secondStage->id)
            ->assertJsonPath('stages.0.color', '#123ABC')
            ->assertJsonPath('stages.1.id', $firstStage->id)
            ->assertJsonPath('stages.1.color', '#654321');

        $this->assertDatabaseHas('deal_stages', [
            'id' => $secondStage->id,
            'sort_order' => 0,
            'color' => '#123ABC',
        ]);

        $this->assertDatabaseHas('deal_stages', [
            'id' => $firstStage->id,
            'sort_order' => 1,
            'color' => '#654321',
        ]);
    }
}
