<?php

namespace Tests\Feature;

use App\Models\Deal;
use App\Models\Pipeline;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CrmFlowTest extends TestCase
{
    use RefreshDatabase;

    public function test_authenticated_user_can_open_core_crm_pages(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $this->actingAs($user)
            ->get(route('dashboard'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('news.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('companies.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('contacts.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('products.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('warehouses.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('tasks.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('calendar.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('projects.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('activities.index'))
            ->assertOk();
    }

    public function test_user_can_create_company_contact_and_deal(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::create([
            'name' => 'Sales',
            'description' => null,
            'is_default' => true,
            'is_active' => true,
            'sort_order' => 0,
        ]);

        $stage = $pipeline->stages()->create([
            'name' => 'Qualification',
            'code' => 'qualification_1',
            'sort_order' => 0,
            'probability' => 30,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $this->actingAs($user)
            ->post(route('companies.store'), [
                'name' => 'Acme Corp',
                'status' => 'lead',
                'owner_id' => $user->id,
                'source' => 'manual',
            ])
            ->assertRedirect();

        $companyId = (int) \App\Models\Company::query()->value('id');

        $this->actingAs($user)
            ->post(route('contacts.store'), [
                'first_name' => 'John',
                'last_name' => 'Doe',
                'email' => 'john.doe@example.com',
                'company_id' => $companyId,
                'owner_id' => $user->id,
                'source' => 'manual',
            ])
            ->assertRedirect();

        $contactId = (int) \App\Models\Contact::query()->value('id');

        $this->actingAs($user)
            ->post(route('deals.store'), [
                'title' => 'Acme annual contract',
                'pipeline_id' => $pipeline->id,
                'stage_id' => $stage->id,
                'company_id' => $companyId,
                'contact_id' => $contactId,
                'owner_id' => $user->id,
                'amount' => 15000,
                'currency' => 'USD',
                'priority' => 'high',
                'status' => 'open',
                'source' => 'manual',
            ])
            ->assertRedirect();

        $this->assertDatabaseHas('deals', [
            'title' => 'Acme annual contract',
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stage->id,
            'status' => 'open',
        ]);
    }

    public function test_user_can_move_deal_stage_and_complete_task(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $pipeline = Pipeline::create([
            'name' => 'Sales',
            'description' => null,
            'is_default' => true,
            'is_active' => true,
            'sort_order' => 0,
        ]);

        $stageA = $pipeline->stages()->create([
            'name' => 'Qualification',
            'code' => 'qualification_1',
            'sort_order' => 0,
            'probability' => 30,
            'is_won' => false,
            'is_lost' => false,
        ]);

        $stageB = $pipeline->stages()->create([
            'name' => 'Won',
            'code' => 'won_2',
            'sort_order' => 1,
            'probability' => 100,
            'is_won' => true,
            'is_lost' => false,
        ]);

        $deal = Deal::create([
            'title' => 'Contract A',
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stageA->id,
            'amount' => 6000,
            'currency' => 'USD',
            'priority' => 'medium',
            'status' => 'open',
            'source' => 'manual',
        ]);

        $task = Task::create([
            'title' => 'Send proposal',
            'deal_id' => $deal->id,
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
            'priority' => 'high',
        ]);

        $this->actingAs($user)
            ->patch(route('deals.update-stage', $deal), [
                'stage_id' => $stageB->id,
            ])
            ->assertRedirect();

        $this->assertDatabaseHas('deals', [
            'id' => $deal->id,
            'stage_id' => $stageB->id,
            'status' => 'won',
        ]);

        $this->actingAs($user)
            ->patch(route('tasks.complete', $task))
            ->assertRedirect();

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'status' => 'done',
        ]);
    }
}
