<?php

namespace Tests\Feature\Api;

use App\Models\UpdateSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;

class UpdatesApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_update_settings_can_be_read_with_updates_read_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.0.0',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        $token = $user->createToken('updates-read', ['updates.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/updates/settings')
            ->assertOk()
            ->assertJsonPath('data.feed_url', 'https://update.crm25.webnet.kz')
            ->assertJsonPath('data.channel', 'stable');
    }

    public function test_update_settings_can_be_updated_with_updates_update_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.0.0',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        $token = $user->createToken('updates-update', ['updates.read', 'updates.update'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/updates/settings', [
                'feed_url' => 'update.crm25.webnet.kz',
                'current_version' => '25.1.0',
                'channel' => 'beta',
                'is_active' => true,
                'auto_check_enabled' => true,
                'check_interval_minutes' => 120,
                'request_timeout_seconds' => 6,
                'verify_tls' => true,
                'meta' => ['region' => 'kz'],
            ])
            ->assertOk()
            ->assertJsonPath('data.feed_url', 'https://update.crm25.webnet.kz')
            ->assertJsonPath('data.current_version', '25.1.0')
            ->assertJsonPath('data.channel', 'beta')
            ->assertJsonPath('data.meta.region', 'kz');
    }

    public function test_update_check_can_be_triggered_with_updates_create_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.0.0',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        Http::fake([
            'https://update.crm25.webnet.kz/api/v1/crm25/updates*' => Http::response([
                'version' => '25.3.0',
                'build' => '250300',
                'is_update_available' => true,
                'notes' => 'Release available.',
            ], 200),
            '*' => Http::response(['message' => 'not used'], 404),
        ]);

        $token = $user->createToken('updates-check', ['updates.read', 'updates.create'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/updates/checks')
            ->assertCreated()
            ->assertJsonPath('data.status', 'success')
            ->assertJsonPath('data.remote_version', '25.3.0')
            ->assertJsonPath('data.is_update_available', true);

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/updates/checks')
            ->assertOk()
            ->assertJsonCount(1, 'data');
    }

    public function test_update_check_is_forbidden_without_updates_create_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        $token = $user->createToken('updates-read-only', ['updates.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/updates/checks')
            ->assertForbidden();
    }
}

