<?php

namespace Tests\Feature\Api;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TaskStatusApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_api_can_update_task_status_for_kanban(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
        ]);

        $token = $user->createToken('mobile-app')->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/tasks/'.$task->id.'/status', [
                'status' => 'review',
            ])
            ->assertOk()
            ->assertJsonPath('data.id', $task->id)
            ->assertJsonPath('data.status', 'review');

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'status' => 'review',
        ]);
    }

    public function test_api_store_subtask_inherits_parent_project(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $user->id,
            'manager_id' => $user->id,
            'status' => 'active',
        ]);

        $parent = Task::factory()->create([
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'project_id' => $project->id,
        ]);

        $token = $user->createToken('mobile-app')->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/tasks', [
                'title' => 'API child',
                'parent_id' => $parent->id,
                'status' => 'todo',
                'priority' => 'medium',
                'estimated_hours' => 2,
                'tracked_hours' => 0,
            ])
            ->assertCreated()
            ->assertJsonPath('data.parent.id', $parent->id)
            ->assertJsonPath('data.project.id', $project->id);

        $this->assertDatabaseHas('tasks', [
            'title' => 'API child',
            'parent_id' => $parent->id,
            'project_id' => $project->id,
        ]);
    }
}
