<?php

namespace Tests\Feature\Api;

use App\Models\Product;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProductApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_read_only_product_token_can_list_products_but_cannot_create(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $product = Product::factory()->create([
            'owner_id' => $user->id,
            'name' => 'Readonly product',
        ]);

        $token = $user->createToken('products-read', ['products.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/products')
            ->assertOk()
            ->assertJsonPath('data.0.id', $product->id);

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/products', [
                'name' => 'Should fail',
                'currency' => 'USD',
                'unit' => 'pcs',
                'status' => 'active',
            ])
            ->assertForbidden();
    }

    public function test_product_crud_token_can_create_update_and_delete_product(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $token = $user->createToken('products-crud', [
            'products.read',
            'products.create',
            'products.update',
            'products.delete',
        ])->plainTextToken;

        $createResponse = $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/products', [
                'name' => 'API Product',
                'sku' => 'API-100',
                'price' => 210.5,
                'currency' => 'usd',
                'unit' => 'pcs',
                'stock' => 25,
                'status' => 'active',
                'owner_id' => $user->id,
            ]);

        $createResponse
            ->assertCreated()
            ->assertJsonPath('data.name', 'API Product')
            ->assertJsonPath('data.currency', 'USD');

        $productId = (int) $createResponse->json('data.id');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/products/'.$productId, [
                'name' => 'API Product Updated',
                'sku' => 'API-100',
                'price' => 190,
                'currency' => 'USD',
                'unit' => 'pcs',
                'stock' => 20,
                'status' => 'archived',
                'owner_id' => $user->id,
            ])
            ->assertOk()
            ->assertJsonPath('data.name', 'API Product Updated')
            ->assertJsonPath('data.status', 'archived');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->deleteJson('/api/v1/products/'.$productId)
            ->assertNoContent();

        $this->assertDatabaseMissing('products', [
            'id' => $productId,
        ]);
    }
}
