<?php

namespace Tests\Feature\Api;

use App\Models\MailServiceSetting;
use App\Models\Mailbox;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class MailApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_mail_settings_can_be_read_with_mail_read_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        MailServiceSetting::query()->create([
            'provider' => 'manual',
            'is_active' => true,
            'domain' => 'mail.crm.local',
            'auto_provision_on_registration' => true,
            'auto_provision_on_user_create' => true,
            'default_status' => 'active',
            'default_quota_mb' => 2048,
        ]);

        $token = $user->createToken('mail-read', ['mail.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/mail/settings')
            ->assertOk()
            ->assertJsonPath('data.domain', 'mail.crm.local');
    }

    public function test_mailbox_crud_works_with_mail_abilities(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        MailServiceSetting::query()->create([
            'provider' => 'manual',
            'is_active' => true,
            'domain' => 'mail.crm.local',
            'auto_provision_on_registration' => true,
            'auto_provision_on_user_create' => true,
            'default_status' => 'active',
            'default_quota_mb' => 2048,
        ]);

        $token = $user->createToken('mail-full', [
            'mail.read',
            'mail.create',
            'mail.update',
            'mail.delete',
        ])->plainTextToken;

        $createResponse = $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/mail/mailboxes', [
                'user_id' => $user->id,
                'local_part' => 'integration.mail',
                'domain' => 'mail.crm.local',
                'status' => 'active',
                'is_primary' => true,
                'quota_mb' => 1024,
                'used_mb' => 128,
            ])
            ->assertCreated()
            ->assertJsonPath('data.address', 'integration.mail@mail.crm.local');

        $mailboxId = (int) $createResponse->json('data.id');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/mail/mailboxes/'.$mailboxId, [
                'user_id' => $user->id,
                'local_part' => 'integration.mail.updated',
                'domain' => 'mail.crm.local',
                'status' => 'suspended',
                'is_primary' => true,
                'quota_mb' => 2048,
                'used_mb' => 512,
            ])
            ->assertOk()
            ->assertJsonPath('data.address', 'integration.mail.updated@mail.crm.local')
            ->assertJsonPath('data.status', 'suspended');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->deleteJson('/api/v1/mail/mailboxes/'.$mailboxId)
            ->assertNoContent();

        $this->assertDatabaseMissing('mailboxes', [
            'id' => $mailboxId,
        ]);
    }

    public function test_mailbox_create_is_forbidden_without_mail_create_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        $token = $user->createToken('mail-read-only', ['mail.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/mail/mailboxes', [
                'user_id' => $user->id,
                'status' => 'active',
            ])
            ->assertForbidden();

        $this->assertSame(0, Mailbox::query()->count());
    }
}
