<?php

namespace Tests\Feature\Api;

use App\Models\HrRequest;
use App\Models\HrSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class HrApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_hr_settings_can_be_read_with_hr_read_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        HrSetting::query()->create([
            'provider' => 'internal',
            'is_active' => true,
            'default_sla_days' => 5,
            'allow_employee_requests' => true,
            'require_approval' => true,
        ]);

        $token = $user->createToken('hr-read', ['hr.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/hr/settings')
            ->assertOk()
            ->assertJsonPath('data.provider', 'internal')
            ->assertJsonPath('data.default_sla_days', 5);
    }

    public function test_hr_request_crud_works_with_hr_abilities(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        $token = $user->createToken('hr-full', [
            'hr.read',
            'hr.create',
            'hr.update',
            'hr.delete',
        ])->plainTextToken;

        $createResponse = $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/hr/requests', [
                'external_id' => 'hr-api-001',
                'type' => 'vacation',
                'status' => 'new',
                'priority' => 'medium',
                'title' => 'Vacation request',
                'description' => 'Need five days off',
                'requested_date' => now()->toDateString(),
                'due_date' => now()->addDays(4)->toDateString(),
                'employee_user_id' => $user->id,
                'assignee_user_id' => $user->id,
                'creator_user_id' => $user->id,
            ])
            ->assertCreated()
            ->assertJsonPath('data.external_id', 'hr-api-001')
            ->assertJsonPath('data.type', 'vacation');

        $requestId = (int) $createResponse->json('data.id');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/hr/requests/'.$requestId, [
                'external_id' => 'hr-api-001',
                'type' => 'vacation',
                'status' => 'in_progress',
                'priority' => 'high',
                'title' => 'Vacation request',
                'description' => 'Need six days off',
                'requested_date' => now()->toDateString(),
                'due_date' => now()->addDays(6)->toDateString(),
                'employee_user_id' => $user->id,
                'assignee_user_id' => $user->id,
                'creator_user_id' => $user->id,
            ])
            ->assertOk()
            ->assertJsonPath('data.status', 'in_progress')
            ->assertJsonPath('data.priority', 'high');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->deleteJson('/api/v1/hr/requests/'.$requestId)
            ->assertNoContent();

        $this->assertDatabaseMissing('hr_requests', [
            'id' => $requestId,
        ]);
    }

    public function test_hr_request_create_is_forbidden_without_hr_create_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        $token = $user->createToken('hr-read-only', ['hr.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/hr/requests', [
                'type' => 'other',
                'status' => 'new',
                'priority' => 'medium',
                'title' => 'Need certificate',
            ])
            ->assertForbidden();

        $this->assertSame(0, HrRequest::query()->count());
    }
}
