<?php

namespace Tests\Feature;

use App\Models\AccessGroup;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class AccessSettingsTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_sees_access_settings_tab_and_can_open_it(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['name' => 'CRM employee']);

        $this->actingAs($admin)
            ->get(route('profile.edit'))
            ->assertOk()
            ->assertSee('Users')
            ->assertSee('Permissions')
            ->assertSee('Modules');

        $this->actingAs($admin)
            ->get(route('profile.edit', ['section' => 'users']))
            ->assertOk()
            ->assertSee('Users')
            ->assertSee('CRM employee');

        $this->actingAs($admin)
            ->get(route('profile.edit', ['section' => 'access']))
            ->assertOk()
            ->assertSee('Permissions')
            ->assertSee('CRM employee')
            ->assertSee('Warehouses')
            ->assertSee('Products')
            ->assertSee('Disk');

        $this->actingAs($admin)
            ->get(route('profile.edit', ['section' => 'modules']))
            ->assertOk()
            ->assertSee('Modules')
            ->assertSee('Upload module archive');
    }

    public function test_russian_locale_translates_users_and_permissions_tabs(): void
    {
        $admin = User::factory()->create([
            'role' => 'admin',
            'locale' => 'ru',
        ]);

        $this->actingAs($admin)
            ->get(route('profile.edit'))
            ->assertOk()
            ->assertSee('Пользователи')
            ->assertSee('Разрешения');
    }

    public function test_non_admin_cannot_update_access_settings(): void
    {
        $moderator = User::factory()->create(['role' => 'manager']);
        $member = User::factory()->create();

        $this->actingAs($moderator)
            ->patch(route('profile.access.update'), [
                'user_id' => $member->id,
                'role' => 'user',
            ])
            ->assertForbidden();
    }

    public function test_non_admin_cannot_impersonate_user(): void
    {
        $moderator = User::factory()->create(['role' => 'manager']);
        $member = User::factory()->create();

        $this->actingAs($moderator)
            ->post(route('profile.access.impersonate', ['member' => $member]))
            ->assertForbidden();
    }

    public function test_admin_can_impersonate_user_from_users_list(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'sales']);

        $this->actingAs($admin)
            ->post(route('profile.access.impersonate', ['member' => $member]))
            ->assertRedirect(route('dashboard'));

        $this->assertAuthenticatedAs($member);
        $this->assertSame($admin->id, session('impersonator_id'));
    }

    public function test_impersonated_user_can_return_to_admin_account(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'sales']);

        $this->actingAs($admin)
            ->post(route('profile.access.impersonate', ['member' => $member]))
            ->assertRedirect(route('dashboard'));

        $this->assertAuthenticatedAs($member);

        $this->actingAs($member)
            ->withSession(['impersonator_id' => $admin->id])
            ->post(route('profile.access.impersonate.leave'))
            ->assertRedirect(route('profile.edit', ['section' => 'users']));

        $this->assertAuthenticatedAs($admin);
        $this->assertNull(session('impersonator_id'));
    }

    public function test_admin_can_update_user_role_and_personalized_permissions(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'sales']);

        $this->actingAs($admin)
            ->patch(route('profile.access.update'), [
                'user_id' => $member->id,
                'role' => 'user',
                'permissions' => [
                    'pipelines' => [
                        'read' => 1,
                        'create' => 1,
                        'update' => 0,
                        'delete' => 0,
                    ],
                ],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $member->refresh();

        $this->assertSame('user', $member->role);
        $this->assertTrue((bool) ($member->permissions['pipelines.create'] ?? false));
        $this->assertFalse((bool) ($member->permissions['pipelines.update'] ?? true));
    }

    public function test_personalized_permission_allows_regular_user_to_create_pipeline(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'user']);

        $pipelinePayload = [
            'name' => 'Personal funnel',
            'sort_order' => 1,
            'is_active' => 1,
            'stages' => "New\nClosed",
        ];

        $this->actingAs($member)
            ->post(route('pipelines.store'), $pipelinePayload)
            ->assertForbidden();

        $this->actingAs($admin)
            ->patch(route('profile.access.update'), [
                'user_id' => $member->id,
                'role' => 'user',
                'permissions' => [
                    'pipelines' => [
                        'read' => 1,
                        'create' => 1,
                        'update' => 0,
                        'delete' => 0,
                    ],
                ],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $member->refresh();

        $this->actingAs($member)
            ->post(route('pipelines.store'), $pipelinePayload)
            ->assertRedirect();

        $this->assertDatabaseHas('pipelines', [
            'name' => 'Personal funnel',
        ]);
    }

    public function test_admin_can_create_group_and_bulk_assign_users(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $first = User::factory()->create(['role' => 'user']);
        $second = User::factory()->create(['role' => 'user']);

        $this->actingAs($admin)
            ->post(route('profile.access.groups.store'), [
                'group_name' => 'Sales Team Group',
                'permissions' => [
                    'deals' => [
                        'read' => 1,
                        'create' => 1,
                    ],
                ],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $group = AccessGroup::query()->where('name', 'Sales Team Group')->firstOrFail();

        $this->actingAs($admin)
            ->patch(route('profile.access.groups.bulk-assign'), [
                'access_group_id' => $group->id,
                'selected_user_ids' => [$first->id, $second->id],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $this->assertSame($group->id, $first->fresh()->access_group_id);
        $this->assertSame($group->id, $second->fresh()->access_group_id);
    }

    public function test_group_permission_allows_regular_user_to_create_pipeline(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'user']);

        $pipelinePayload = [
            'name' => 'Group enabled funnel',
            'sort_order' => 1,
            'is_active' => 1,
            'stages' => "New\nClosed",
        ];

        $this->actingAs($member)
            ->post(route('pipelines.store'), $pipelinePayload)
            ->assertForbidden();

        $this->actingAs($admin)
            ->post(route('profile.access.groups.store'), [
                'group_name' => 'Pipeline Creators',
                'permissions' => [
                    'pipelines' => [
                        'read' => 1,
                        'create' => 1,
                        'update' => 0,
                        'delete' => 0,
                    ],
                ],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $group = AccessGroup::query()->where('name', 'Pipeline Creators')->firstOrFail();

        $this->actingAs($admin)
            ->patch(route('profile.access.groups.bulk-assign'), [
                'access_group_id' => $group->id,
                'selected_user_ids' => [$member->id],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $member->refresh();

        $this->actingAs($member)
            ->post(route('pipelines.store'), $pipelinePayload)
            ->assertRedirect();

        $this->assertDatabaseHas('pipelines', [
            'name' => 'Group enabled funnel',
        ]);
    }

    public function test_personal_permission_overrides_group_permission(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'user']);

        $this->actingAs($admin)
            ->post(route('profile.access.groups.store'), [
                'group_name' => 'Pipelines Allowed',
                'permissions' => [
                    'pipelines' => [
                        'read' => 1,
                        'create' => 1,
                    ],
                ],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $group = AccessGroup::query()->where('name', 'Pipelines Allowed')->firstOrFail();

        $this->actingAs($admin)
            ->patch(route('profile.access.groups.bulk-assign'), [
                'access_group_id' => $group->id,
                'selected_user_ids' => [$member->id],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $this->actingAs($admin)
            ->patch(route('profile.access.update'), [
                'user_id' => $member->id,
                'role' => 'user',
                'access_group_id' => $group->id,
                'permissions' => [
                    'pipelines' => [
                        'read' => 1,
                        'create' => 0,
                        'update' => 0,
                        'delete' => 0,
                    ],
                ],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'access']));

        $member->refresh();

        $this->actingAs($member)
            ->post(route('pipelines.store'), [
                'name' => 'Should stay forbidden',
                'sort_order' => 1,
                'is_active' => 1,
                'stages' => "New\nClosed",
            ])
            ->assertForbidden();
    }
}
