@php
    $baseQuery = request()->except(['view', 'page']);
    $statusLabels = $statusOptions ?? [];
    $priorityLabels = $priorityOptions ?? [];
    $kanbanStageRows = old('stages', $kanbanStages ?? []);
    $advancedFiltersCount = collect([
        $priority !== null && $priority !== '',
        $creatorId > 0,
        $projectId > 0,
        $deadline !== null && $deadline !== '',
        $parentMode !== 'all',
        $dateFrom !== '',
        $dateTo !== '',
        $mine,
    ])->filter()->count();
    $hasAdvancedFilters = $advancedFiltersCount > 0;
@endphp

<x-app-layout>
    <x-slot name="header">
        <div class="flex flex-wrap items-center justify-between gap-3">
            <div>
                <h2 class="font-semibold text-xl text-gray-800 leading-tight">Tasks</h2>
                <p class="text-sm text-gray-500">Task work area: list, kanban, Gantt chart, subtasks</p>
            </div>
            <div class="flex items-center gap-2">
                @include('partials.section-access-controls', [
                    'sectionAccessEntity' => 'tasks',
                    'sectionAccessTitle' => __('Tasks'),
                    'canManageSectionAccess' => $canManageSectionAccess ?? false,
                    'sectionAccessUsers' => $sectionAccessUsers ?? collect(),
                ])
                <a href="{{ route('tasks.create') }}" class="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white hover:bg-indigo-500">New task</a>
            </div>
        </div>
    </x-slot>

    <div class="pb-12" data-sidepanel-scope>
        <div class="w-full px-6 space-y-4">
            <section class="bg-white border border-gray-200 rounded-xl p-4 space-y-4">
                <div class="flex flex-wrap gap-2">
                    @foreach ($viewOptions as $mode => $label)
                        <a href="{{ route('tasks.index', array_merge($baseQuery, ['view' => $mode])) }}"
                           class="inline-flex items-center rounded-md px-3 py-1.5 text-sm font-medium {{ $viewMode === $mode ? 'bg-indigo-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200' }}">
                            {{ $label }}
                        </a>
                    @endforeach
                </div>

                <form method="GET" action="{{ route('tasks.index') }}" class="space-y-3">
                    <input type="hidden" name="view" value="{{ $viewMode }}">

                    <div class="grid grid-cols-1 gap-3 md:grid-cols-2 xl:grid-cols-5">
                        <div class="xl:col-span-2">
                            <label for="q" class="block text-xs font-medium text-gray-500 uppercase">Title contains</label>
                            <input id="q" type="text" name="q" value="{{ $search }}" placeholder="Task title" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>

                        <div>
                            <label for="status" class="block text-xs font-medium text-gray-500 uppercase">Status</label>
                            <select id="status" name="status" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                <option value="">All</option>
                                @foreach ($statusOptions as $value => $label)
                                    <option value="{{ $value }}" @selected($status === $value)>{{ $label }}</option>
                                @endforeach
                            </select>
                        </div>

                        <div>
                            <label for="assignee_id" class="block text-xs font-medium text-gray-500 uppercase">Executor</label>
                            <x-user-search-select
                                id="assignee_id"
                                name="assignee_id"
                                :users="$users"
                                :selected="$assigneeId"
                                placeholder="All users"
                                empty-label="All"
                            />
                        </div>

                        <div class="flex items-end gap-2">
                            <button type="submit" class="inline-flex items-center rounded-md bg-gray-800 px-4 py-2 text-sm font-medium text-white hover:bg-gray-700">Apply</button>
                            <a href="{{ route('tasks.index', ['view' => $viewMode]) }}" class="inline-flex items-center rounded-md border border-gray-300 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">Reset</a>
                        </div>
                    </div>

                    <details
                        class="rounded-lg border border-gray-200 bg-gray-50 p-3"
                        data-advanced-task-filters
                        data-advanced-open="{{ $hasAdvancedFilters ? '1' : '0' }}"
                        @if ($hasAdvancedFilters) open @endif
                    >
                        <summary class="cursor-pointer select-none text-sm font-medium text-gray-700">
                            More filters
                            @if ($advancedFiltersCount > 0)
                                <span class="ml-2 inline-flex rounded-full bg-indigo-100 px-2 py-0.5 text-xs font-semibold text-indigo-700">{{ $advancedFiltersCount }}</span>
                            @endif
                        </summary>

                        <div class="mt-3 grid grid-cols-1 gap-3 md:grid-cols-2 xl:grid-cols-4">
                            <div>
                                <label for="priority" class="block text-xs font-medium text-gray-500 uppercase">Priority</label>
                                <select id="priority" name="priority" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                    <option value="">All</option>
                                    @foreach ($priorityOptions as $value => $label)
                                        <option value="{{ $value }}" @selected($priority === $value)>{{ $label }}</option>
                                    @endforeach
                                </select>
                            </div>

                            <div>
                                <label for="creator_id" class="block text-xs font-medium text-gray-500 uppercase">Stage Director</label>
                                <x-user-search-select
                                    id="creator_id"
                                    name="creator_id"
                                    :users="$users"
                                    :selected="$creatorId"
                                    placeholder="All users"
                                    empty-label="All"
                                />
                            </div>

                            <div>
                                <label for="project_id" class="block text-xs font-medium text-gray-500 uppercase">Project</label>
                                <select id="project_id" name="project_id" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                    <option value="">All</option>
                                    @foreach ($projects as $project)
                                        <option value="{{ $project->id }}" @selected($projectId === $project->id)>{{ $project->name }}</option>
                                    @endforeach
                                </select>
                            </div>

                            <div>
                                <label for="deadline" class="block text-xs font-medium text-gray-500 uppercase">Term</label>
                                <select id="deadline" name="deadline" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                    <option value="">Any</option>
                                    <option value="overdue" @selected($deadline === 'overdue')>Overdue</option>
                                    <option value="today" @selected($deadline === 'today')>Today</option>
                                    <option value="week" @selected($deadline === 'week')>During the week</option>
                                    <option value="no_deadline" @selected($deadline === 'no_deadline')>No deadline</option>
                                </select>
                            </div>

                            <div>
                                <label for="parent_mode" class="block text-xs font-medium text-gray-500 uppercase">Structure</label>
                                <select id="parent_mode" name="parent_mode" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                    @foreach ($parentModeOptions as $value => $label)
                                        <option value="{{ $value }}" @selected($parentMode === $value)>{{ $label }}</option>
                                    @endforeach
                                </select>
                            </div>

                            <div>
                                <label for="date_from" class="block text-xs font-medium text-gray-500 uppercase">Term from</label>
                                <input id="date_from" type="date" name="date_from" value="{{ $dateFrom }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            </div>

                            <div>
                                <label for="date_to" class="block text-xs font-medium text-gray-500 uppercase">Deadline by</label>
                                <input id="date_to" type="date" name="date_to" value="{{ $dateTo }}" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            </div>

                            <div class="flex items-end">
                                <label class="inline-flex items-center gap-2 text-sm text-gray-700">
                                    <input type="checkbox" name="mine" value="1" @checked($mine) class="rounded border-gray-300 text-indigo-600 shadow-sm focus:ring-indigo-500">
                                    Only mine
                                </label>
                            </div>
                        </div>
                    </details>
                </form>
            </section>

            @if ($viewMode === 'kanban')
                <section id="task-kanban" data-csrf="{{ csrf_token() }}" data-kanban-preset-url="{{ route('tasks.kanban.preset') }}" class="space-y-3">
                    <div class="text-xs text-gray-500">Drag-and-drop between statuses + real-time updates</div>

                    <div class="relative" data-board-scroll-wrap>
                    <div class="overflow-x-auto pb-2" data-board-scroll>
                    <div class="flex min-w-max items-start gap-4" data-board-grid>
                        @foreach ($kanbanColumns as $column)
                            <div
                                class="w-[320px] shrink-0 self-start rounded-xl border border-gray-200 border-t-4 bg-white p-3 flex flex-col"
                                data-status-column="{{ $column['status'] }}"
                                data-stage-color="{{ $column['color'] }}"
                                style="border-top-color: {{ $column['color'] }}; background-image: linear-gradient(to bottom, {{ $column['color'] }}1F, #FFFFFF 42%);"
                            >
                                <div class="mb-3 flex items-start justify-between gap-2">
                                    <div class="min-w-0">
                                        <p class="truncate font-semibold text-gray-900" data-stage-label-text>{{ $column['label'] }}</p>
                                        <p class="text-xs text-gray-500"><span data-stage-count>{{ $column['count'] }}</span> tasks · <span data-stage-hours>{{ number_format((float) $column['hours'], 1, '.', ' ') }}</span>h</p>
                                    </div>
                                    <div class="flex items-center gap-1">
                                        <label class="inline-flex h-7 w-7 cursor-pointer items-center justify-center rounded-md border border-gray-200 text-gray-500 hover:bg-gray-50" title="Stage color">
                                            <x-menu-icon name="fa-solid fa-palette" class="h-3.5 w-3.5" />
                                            <input
                                                type="color"
                                                value="{{ preg_match('/^#[0-9A-Fa-f]{6}$/', (string) $column['color']) ? strtoupper((string) $column['color']) : '#94A3B8' }}"
                                                class="sr-only"
                                                data-stage-color-picker
                                            >
                                        </label>
                                        <button
                                            type="button"
                                            class="inline-flex h-7 w-7 cursor-grab items-center justify-center rounded-md border border-gray-200 text-gray-500 hover:bg-gray-50 active:cursor-grabbing"
                                            title="Move Stage"
                                            data-stage-move-handle
                                            draggable="true"
                                        >
                                            <x-menu-icon name="fa-solid fa-grip-vertical" class="h-3.5 w-3.5" />
                                        </button>
                                    </div>
                                </div>

                                <div class="space-y-2 max-h-[36rem] overflow-auto pr-1" data-stage-list>
                                    @forelse ($column['tasks'] as $task)
                                        <article class="h-auto rounded-lg border border-gray-100 p-3 bg-gray-50 cursor-move {{ $task->status === 'done' ? 'opacity-75' : '' }}"
                                                 draggable="true"
                                                 data-task-card
                                                 data-task-id="{{ $task->id }}"
                                                 data-task-status="{{ $task->status }}"
                                                 data-task-hours="{{ (float) $task->tracked_hours }}"
                                                 data-update-url="{{ route('tasks.update-status', $task) }}">
                                            <div class="flex items-start justify-between gap-2">
                                                <a href="{{ route('tasks.show', $task) }}" class="text-sm font-medium text-gray-900 hover:text-indigo-600" data-task-title>#{{ $task->id }} · {{ $task->title }}</a>
                                                <span class="inline-flex rounded-full px-2 py-0.5 text-[11px] font-medium bg-gray-200 text-gray-700" data-task-priority>{{ $priorityLabels[$task->priority] ?? $task->priority }}</span>
                                            </div>

                                            <p class="mt-1 text-xs text-gray-500" data-assignee-name>{{ $task->assignee?->name ?? 'Without performer' }}</p>
                                            <p class="text-xs text-gray-500" data-due-date>{{ $task->due_at?->format('d.m H:i') ?? 'No deadline' }}</p>
                                            <p class="text-xs text-gray-500" data-parent-title>{{ $task->parent?->title ? 'Subtask: '.$task->parent->title : 'Main task' }}</p>
                                            <p class="text-xs text-gray-500" data-subtask-progress>Subtasks: {{ $task->done_subtasks_count }}/{{ $task->subtasks_count }}</p>

                                            <div class="mt-2 flex items-center justify-between text-xs text-gray-500">
                                                <span>Fact: <span data-tracked-hours>{{ number_format((float) $task->tracked_hours, 1, '.', ' ') }}</span>h</span>
                                                <a href="{{ route('tasks.show', $task) }}" data-edit-link class="text-indigo-600 hover:text-indigo-500">Open</a>
                                            </div>

                                            <div class="mt-2">
                                                <select class="w-full rounded-md border-gray-300 text-xs shadow-sm focus:border-indigo-500 focus:ring-indigo-500" data-status-select>
                                                    @foreach ($statusOptions as $statusValue => $statusLabel)
                                                        <option value="{{ $statusValue }}" @selected($task->status === $statusValue)>{{ $statusLabel }}</option>
                                                    @endforeach
                                                </select>
                                            </div>
                                        </article>
                                    @empty
                                        <p class="text-xs text-gray-500" data-empty>No tasks</p>
                                    @endforelse
                                </div>
                            </div>
                        @endforeach
                    </div>
                    </div>
                    <div
                        class="pointer-events-none absolute inset-y-0 right-0 w-16 bg-gradient-to-l from-gray-100 via-gray-100/90 to-transparent opacity-0 transition-opacity duration-200"
                        data-board-scroll-fade-right
                    ></div>
                    </div>
                </section>
            @elseif ($viewMode === 'gantt')
                <section class="bg-white border border-gray-200 rounded-xl p-4">
                    @if ($ganttRows->isEmpty() || ! $ganttPeriod)
                        <p class="text-sm text-gray-500">There is no data for the Gantt chart. Add "Start" and "Due" fields to tasks.</p>
                    @else
                        <div class="mb-4 text-sm text-gray-600">
                            Period: {{ $ganttPeriod['start']->format('d.m.Y') }} - {{ $ganttPeriod['end']->format('d.m.Y') }}
                        </div>

                        <div class="space-y-2">
                            @foreach ($ganttRows as $row)
                                <div class="grid grid-cols-1 xl:grid-cols-12 gap-2 items-center">
                                    <div class="xl:col-span-4 rounded-lg border border-gray-100 p-2 bg-gray-50">
                                        <a href="{{ route('tasks.show', $row['task']) }}" class="text-sm font-medium text-gray-900 hover:text-indigo-600">#{{ $row['task']->id }} · {{ $row['task']->title }}</a>
                                        <p class="text-xs text-gray-500">{{ $row['task']->assignee?->name ?? 'Without performer' }} · {{ $statusLabels[$row['task']->status] ?? $row['task']->status }}</p>
                                        <p class="text-xs text-gray-500">{{ $row['start']->format('d.m') }} - {{ $row['end']->format('d.m') }}</p>
                                    </div>
                                    <div class="xl:col-span-8 relative h-10 rounded-lg bg-gray-50 border border-gray-100 overflow-hidden">
                                        <div class="absolute inset-y-0 w-px bg-red-300" style="left: {{ $ganttPeriod['today_position'] }}%;"></div>
                                        <a href="{{ route('tasks.show', $row['task']) }}"
                                           class="absolute top-1 h-8 rounded-md px-2 flex items-center text-xs font-medium text-white {{ $row['is_overdue'] ? 'bg-red-500' : ($row['task']->status === 'done' ? 'bg-emerald-500' : 'bg-indigo-500') }}"
                                           style="left: {{ $row['left'] }}%; width: {{ $row['width'] }}%;">
                                            {{ $priorityLabels[$row['task']->priority] ?? $row['task']->priority }}
                                        </a>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @endif
                </section>
            @else
                <section class="bg-white border border-gray-200 rounded-xl overflow-hidden">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Task</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Performer/Producer</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Period</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Priority</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Structure</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Connection</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100 bg-white">
                            @forelse ($tasks as $task)
                                <tr class="hover:bg-gray-50">
                                    <td class="px-4 py-3 align-top">
                                        <a href="{{ route('tasks.show', $task) }}" class="font-medium text-gray-900 hover:text-indigo-600">#{{ $task->id }} · {{ $task->title }}</a>
                                        <div class="text-xs text-gray-500">Plan/actual: {{ number_format((float) $task->estimated_hours, 1, '.', ' ') }}h / {{ number_format((float) $task->tracked_hours, 1, '.', ' ') }}h</div>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        <div>{{ $task->assignee?->name ?? '—' }}</div>
                                        <div class="text-xs text-gray-500">{{ $task->creator?->name ?? '—' }}</div>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        <div>{{ $task->starts_at?->format('d.m H:i') ?? '—' }}</div>
                                        <div class="text-xs text-gray-500">to {{ $task->due_at?->format('d.m H:i') ?? '—' }}</div>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">{{ $priorityLabels[$task->priority] ?? $task->priority }}</td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        <div>
                                            @if ($task->parent)
                                                <a href="{{ route('tasks.show', $task->parent) }}" class="text-indigo-600 hover:text-indigo-500 text-xs">Subtask #{{ $task->parent->id }}</a>
                                            @else
                                                <span class="text-xs text-gray-500">Main</span>
                                            @endif
                                        </div>
                                        <div class="text-xs text-gray-500">{{ $task->done_subtasks_count }}/{{ $task->subtasks_count }} subtasks</div>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">{{ $task->project?->name ?? $task->deal?->title ?? $task->company?->name ?? $task->contact?->full_name ?? '—' }}</td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">{{ $statusLabels[$task->status] ?? $task->status }}</td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        <div class="flex items-center gap-2">
                                            <a href="{{ route('tasks.show', $task) }}" class="text-indigo-600 hover:text-indigo-500">Open</a>
                                            @if ($task->status !== 'done')
                                                <form method="POST" action="{{ route('tasks.complete', $task) }}">
                                                    @csrf
                                                    @method('PATCH')
                                                    <button type="submit" class="text-emerald-600 hover:text-emerald-500">Ready</button>
                                                </form>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="8" class="px-4 py-8 text-center text-sm text-gray-500">No tasks found.</td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </section>

                <div>{{ $tasks?->links() }}</div>
            @endif
        </div>
    </div>

    @if ($viewMode === 'kanban')
        <script>
            document.addEventListener('DOMContentLoaded', () => {
                const settingsForm = document.querySelector('[data-kanban-stage-settings]');
                const settingsList = settingsForm?.querySelector('[data-stage-settings-list]');
                const hiddenInputsNode = settingsForm?.querySelector('[data-stage-hidden-inputs]');
                const addStageButton = settingsForm?.querySelector('[data-add-stage]');
                let draggingStageRow = null;

                const slugifyStatus = (value) => {
                    const base = String(value || '')
                        .toLowerCase()
                        .replace(/[^a-z0-9]+/g, '_')
                        .replace(/^_+|_+$/g, '');

                    return base;
                };

                const nextCustomStatus = () => {
                    return `custom_stage_${Date.now().toString(36)}_${Math.floor(Math.random() * 9999)}`;
                };

                const stageRows = () => {
                    return settingsList ? Array.from(settingsList.querySelectorAll('[data-stage-row]')) : [];
                };

                const renderHiddenInputs = () => {
                    if (!hiddenInputsNode) {
                        return;
                    }

                    hiddenInputsNode.innerHTML = '';

                    stageRows().forEach((row, index) => {
                        const statusInput = row.querySelector('[data-stage-status]');
                        const labelInput = row.querySelector('[data-stage-label]');
                        const colorInput = row.querySelector('[data-stage-color]');
                        const coreInput = row.querySelector('[data-stage-is-core]');

                        const stageData = [
                            [`stages[${index}][status]`, statusInput?.value || ''],
                            [`stages[${index}][label]`, labelInput?.value || ''],
                            [`stages[${index}][color]`, colorInput?.value || '#94A3B8'],
                            [`stages[${index}][is_core]`, coreInput?.value || '0'],
                        ];

                        stageData.forEach(([name, value]) => {
                            const input = document.createElement('input');
                            input.type = 'hidden';
                            input.name = name;
                            input.value = value;
                            hiddenInputsNode.appendChild(input);
                        });
                    });
                };

                const normalizeStageRows = () => {
                    const used = new Set();

                    stageRows().forEach((row) => {
                        const statusInput = row.querySelector('[data-stage-status]');
                        const labelInput = row.querySelector('[data-stage-label]');
                        const codeNode = row.querySelector('[data-stage-code]');
                        const coreInput = row.querySelector('[data-stage-is-core]');
                        const isCore = coreInput?.value === '1';

                        if (!statusInput || !labelInput) {
                            return;
                        }

                        let status = String(statusInput.value || '').trim().toLowerCase();

                        if (!isCore) {
                            const fromLabel = slugifyStatus(labelInput.value);
                            if (!status || status.startsWith('custom_stage_') || status.startsWith('custom_new_')) {
                                status = fromLabel ? `custom_${fromLabel}` : nextCustomStatus();
                            }

                            status = slugifyStatus(status);
                            if (!status.startsWith('custom_')) {
                                status = `custom_${status}`;
                            }
                        }

                        if (!status) {
                            status = isCore ? 'todo' : nextCustomStatus();
                        }

                        let uniqueStatus = status;
                        let suffix = 2;
                        while (used.has(uniqueStatus)) {
                            uniqueStatus = `${status}_${suffix}`;
                            suffix += 1;
                        }

                        used.add(uniqueStatus);
                        statusInput.value = uniqueStatus;
                        if (codeNode) {
                            codeNode.textContent = uniqueStatus;
                        }
                    });

                    renderHiddenInputs();
                };

                const createStageRow = ({ status, label, color }) => {
                    const row = document.createElement('div');
                    row.className = 'grid grid-cols-[auto_1fr_auto] items-start gap-3 border-b border-gray-100 bg-white px-3 py-3 last:border-b-0';
                    row.setAttribute('data-stage-row', '');
                    row.setAttribute('data-stage-core', '0');
                    row.setAttribute('draggable', 'true');

                    row.innerHTML = `
                        <button type="button" class="mt-7 inline-flex h-8 w-8 items-center justify-center rounded-md border border-gray-200 text-gray-500 hover:bg-gray-50" title="Drag stage" data-drag-handle>
                            <i class="fa-solid fa-grip-vertical text-sm" aria-hidden="true"></i>
                        </button>
                        <div class="grid grid-cols-1 gap-3 md:grid-cols-[minmax(0,1fr)_160px]">
                            <div>
                                <label class="block text-xs font-medium uppercase tracking-wide text-gray-500">Name</label>
                                <input type="text" value="${label}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" maxlength="60" required data-stage-label>
                                <input type="hidden" value="${status}" data-stage-status>
                                <input type="hidden" value="0" data-stage-is-core>
                                <p class="mt-1 text-[11px] text-gray-400" data-stage-code>${status}</p>
                            </div>
                            <div>
                                <label class="block text-xs font-medium uppercase tracking-wide text-gray-500">
                                    <span class="inline-flex items-center gap-1">
                                        <i class="fa-solid fa-palette text-xs" aria-hidden="true"></i>
                                        Color
                                    </span>
                                </label>
                                <input type="color" value="${color}" class="mt-1 h-10 w-full rounded-md border border-gray-300 bg-white px-1 py-1" data-stage-color>
                            </div>
                        </div>
                        <button type="button" class="mt-7 inline-flex h-8 w-8 items-center justify-center rounded-md border border-gray-200 text-gray-500 hover:border-red-300 hover:text-red-600" data-remove-stage title="Delete stage">
                            <i class="fa-solid fa-trash-can text-sm" aria-hidden="true"></i>
                        </button>
                    `;

                    return row;
                };

                if (settingsForm && settingsList && hiddenInputsNode) {
                    settingsList.addEventListener('input', (event) => {
                        const target = event.target;
                        if (!(target instanceof HTMLElement)) {
                            return;
                        }

                        if (target.matches('[data-stage-label], [data-stage-color]')) {
                            normalizeStageRows();
                        }
                    });

                    settingsList.addEventListener('click', (event) => {
                        const target = event.target instanceof HTMLElement ? event.target : null;
                        const removeButton = target?.closest('[data-remove-stage]');
                        if (!removeButton) {
                            return;
                        }

                        const row = removeButton.closest('[data-stage-row]');
                        if (!row) {
                            return;
                        }

                        const isCore = row.querySelector('[data-stage-is-core]')?.value === '1';
                        if (isCore) {
                            return;
                        }

                        row.remove();
                        normalizeStageRows();
                    });

                    settingsList.addEventListener('dragstart', (event) => {
                        const target = event.target instanceof HTMLElement ? event.target : null;
                        const row = target?.closest('[data-stage-row]');
                        if (!row) {
                            return;
                        }

                        const isHandle = !!target?.closest('[data-drag-handle]');
                        if (!isHandle) {
                            event.preventDefault();
                            return;
                        }

                        draggingStageRow = row;
                        row.classList.add('opacity-60');

                        if (event.dataTransfer) {
                            event.dataTransfer.effectAllowed = 'move';
                            event.dataTransfer.setData('text/plain', 'stage-row');
                        }
                    });

                    settingsList.addEventListener('dragend', () => {
                        stageRows().forEach((row) => row.classList.remove('opacity-60'));
                        draggingStageRow = null;
                        normalizeStageRows();
                    });

                    settingsList.addEventListener('dragover', (event) => {
                        if (!draggingStageRow) {
                            return;
                        }

                        const target = event.target instanceof HTMLElement
                            ? event.target.closest('[data-stage-row]')
                            : null;

                        if (!target || target === draggingStageRow) {
                            return;
                        }

                        event.preventDefault();
                        const rect = target.getBoundingClientRect();
                        const shouldInsertAfter = event.clientY > rect.top + rect.height / 2;

                        if (shouldInsertAfter) {
                            settingsList.insertBefore(draggingStageRow, target.nextElementSibling);
                        } else {
                            settingsList.insertBefore(draggingStageRow, target);
                        }
                    });

                    addStageButton?.addEventListener('click', () => {
                        const row = createStageRow({
                            status: nextCustomStatus(),
                            label: 'New stage',
                            color: '#6366F1',
                        });

                        settingsList.appendChild(row);
                        normalizeStageRows();
                        const labelInput = row.querySelector('[data-stage-label]');
                        if (labelInput instanceof HTMLInputElement) {
                            labelInput.focus();
                            labelInput.select();
                        }
                    });

                    settingsForm.addEventListener('submit', () => {
                        normalizeStageRows();
                    });

                    normalizeStageRows();
                }

                const board = document.getElementById('task-kanban');
                if (!board) {
                    return;
                }

                const csrf = board.dataset.csrf;
                const boardGrid = board.querySelector('[data-board-grid]');
                if (!boardGrid) {
                    return;
                }

                const getStatusColumns = () => Array.from(board.querySelectorAll('[data-status-column]'));
                const coreStatuses = ['todo', 'in_progress', 'review', 'done'];
                let draggingCard = null;
                let draggingStageColumn = null;
                let stagePresetSaveTimer = null;
                const priorityLabels = @json($priorityLabels);

                const hoursFormat = (value) => {
                    const numeric = Number(value || 0);
                    return new Intl.NumberFormat(document.documentElement.lang || 'ru-RU', {
                        minimumFractionDigits: 1,
                        maximumFractionDigits: 1,
                    }).format(numeric);
                };

                const formatDue = (isoDate) => {
                    if (!isoDate) {
                        return 'No deadline';
                    }

                    const date = new Date(isoDate);
                    if (Number.isNaN(date.getTime())) {
                        return 'No deadline';
                    }

                    return new Intl.DateTimeFormat('ru-RU', {
                        day: '2-digit',
                        month: '2-digit',
                        hour: '2-digit',
                        minute: '2-digit',
                    }).format(date);
                };

                const normalizeStageColor = (rawColor) => {
                    const color = String(rawColor || '').trim().toUpperCase();

                    if (!/^#[0-9A-F]{6}$/.test(color)) {
                        return '#94A3B8';
                    }

                    return color;
                };

                const applyColumnColor = (column, rawColor) => {
                    const color = normalizeStageColor(rawColor);
                    column.dataset.stageColor = color;
                    column.style.borderTopColor = color;
                    column.style.backgroundImage = `linear-gradient(to bottom, ${color}1F, #FFFFFF 42%)`;

                    const picker = column.querySelector('[data-stage-color-picker]');
                    if (picker) {
                        picker.value = color;
                    }
                };

                const syncSettingsRowsFromBoard = () => {
                    if (!settingsList || !hiddenInputsNode) {
                        return;
                    }

                    const rowsByStatus = new Map(
                        stageRows()
                            .map((row) => {
                                const statusInput = row.querySelector('[data-stage-status]');
                                if (!statusInput) {
                                    return null;
                                }

                                return [String(statusInput.value || ''), row];
                            })
                            .filter(Boolean)
                    );

                    getStatusColumns().forEach((column) => {
                        const status = String(column.dataset.statusColumn || '');
                        const row = rowsByStatus.get(status);
                        if (!row) {
                            return;
                        }

                        const labelText = column.querySelector('[data-stage-label-text]')?.textContent?.trim() || status;
                        const colorValue = normalizeStageColor(column.dataset.stageColor || '#94A3B8');

                        const labelInput = row.querySelector('[data-stage-label]');
                        if (labelInput instanceof HTMLInputElement) {
                            labelInput.value = labelText;
                        }

                        const colorInput = row.querySelector('[data-stage-color]');
                        if (colorInput instanceof HTMLInputElement) {
                            colorInput.value = colorValue;
                        }

                        settingsList.appendChild(row);
                    });

                    normalizeStageRows();
                };

                const collectStagePresetFromBoard = () => {
                    return getStatusColumns().map((column) => {
                        const status = String(column.dataset.statusColumn || '');
                        const label = column.querySelector('[data-stage-label-text]')?.textContent?.trim() || status;
                        const color = normalizeStageColor(column.dataset.stageColor || '#94A3B8');

                        return {
                            status,
                            label,
                            color,
                            is_core: coreStatuses.includes(status) ? 1 : 0,
                        };
                    });
                };

                const persistStagePreset = async () => {
                    const actionUrl = board.dataset.kanbanPresetUrl || settingsForm?.getAttribute('action');
                    if (!actionUrl) {
                        return;
                    }

                    try {
                        await fetch(actionUrl, {
                            method: 'PATCH',
                            headers: {
                                'Content-Type': 'application/json',
                                'Accept': 'application/json',
                                'X-CSRF-TOKEN': csrf,
                                'X-Requested-With': 'XMLHttpRequest',
                            },
                            body: JSON.stringify({ stages: collectStagePresetFromBoard() }),
                        });
                    } catch (error) {
                        // Keep UI responsive on network errors.
                    }
                };

                const queueStagePresetSave = () => {
                    if (stagePresetSaveTimer) {
                        window.clearTimeout(stagePresetSaveTimer);
                    }

                    stagePresetSaveTimer = window.setTimeout(() => {
                        persistStagePreset();
                    }, 180);
                };

                const updateColumnStats = (column) => {
                    const list = column.querySelector('[data-stage-list]');
                    const cards = list ? Array.from(list.querySelectorAll('[data-task-card]')) : [];
                    const count = cards.length;
                    const hours = cards.reduce((sum, card) => sum + Number(card.dataset.taskHours || 0), 0);

                    const countNode = column.querySelector('[data-stage-count]');
                    const hoursNode = column.querySelector('[data-stage-hours]');
                    if (countNode) {
                        countNode.textContent = String(count);
                    }
                    if (hoursNode) {
                        hoursNode.textContent = hoursFormat(hours);
                    }

                    const emptyNode = list ? list.querySelector('[data-empty]') : null;
                    if (emptyNode) {
                        emptyNode.classList.toggle('hidden', count > 0);
                    }

                    if (count === 0 && list && !emptyNode) {
                        const placeholder = document.createElement('p');
                        placeholder.dataset.empty = '1';
                        placeholder.className = 'text-xs text-gray-500';
                        placeholder.textContent = 'No tasks';
                        list.appendChild(placeholder);
                    }
                };

                const updateAllStats = () => {
                    getStatusColumns().forEach(updateColumnStats);
                };

                const placeCardIntoStatus = (card, status) => {
                    const column = board.querySelector(`[data-status-column="${status}"]`);
                    const list = column ? column.querySelector('[data-stage-list]') : null;

                    if (!list) {
                        return;
                    }

                    const emptyNode = list.querySelector('[data-empty]');
                    if (emptyNode) {
                        emptyNode.remove();
                    }

                    list.prepend(card);
                    card.dataset.taskStatus = status;

                    const select = card.querySelector('[data-status-select]');
                    if (select) {
                        select.value = status;
                    }

                    card.classList.toggle('opacity-75', status === 'done');
                    updateAllStats();
                };

                const syncTaskFromPayload = (task) => {
                    const card = board.querySelector(`[data-task-card][data-task-id="${task.id}"]`);
                    if (!card) {
                        return;
                    }

                    card.dataset.taskHours = String(task.tracked_hours || 0);

                    const titleNode = card.querySelector('[data-task-title]');
                    const assigneeNode = card.querySelector('[data-assignee-name]');
                    const dueNode = card.querySelector('[data-due-date]');
                    const priorityNode = card.querySelector('[data-task-priority]');
                    const parentNode = card.querySelector('[data-parent-title]');
                    const progressNode = card.querySelector('[data-subtask-progress]');
                    const trackedNode = card.querySelector('[data-tracked-hours]');
                    const editNode = card.querySelector('[data-edit-link]');

                    if (titleNode && task.title) {
                        titleNode.textContent = `#${task.id} · ${task.title}`;
                        if (task.url) {
                            titleNode.href = task.url;
                        }
                    }
                    if (assigneeNode) {
                        assigneeNode.textContent = task.assignee_name || 'Without performer';
                    }
                    if (dueNode) {
                        dueNode.textContent = formatDue(task.due_at);
                    }
                    if (priorityNode) {
                        priorityNode.textContent = priorityLabels[task.priority] || task.priority || '';
                    }
                    if (parentNode) {
                        parentNode.textContent = task.parent_title ? `Subtask: ${task.parent_title}` : 'Main task';
                    }
                    if (progressNode) {
                        progressNode.textContent = `Subtasks: ${task.done_subtasks_count || 0}/${task.subtasks_count || 0}`;
                    }
                    if (trackedNode) {
                        trackedNode.textContent = hoursFormat(task.tracked_hours || 0);
                    }
                    if (editNode && task.edit_url) {
                        editNode.href = task.edit_url;
                    }

                    placeCardIntoStatus(card, task.status);
                };

                const updateTaskStatus = async (card, status, previousStatus = card.dataset.taskStatus) => {
                    const url = card.dataset.updateUrl;

                    try {
                        const response = await fetch(url, {
                            method: 'PATCH',
                            headers: {
                                'Content-Type': 'application/json',
                                'Accept': 'application/json',
                                'X-CSRF-TOKEN': csrf,
                            },
                            body: JSON.stringify({ status }),
                        });

                        if (!response.ok) {
                            throw new Error('Task status update failed');
                        }

                        const data = await response.json();
                        if (!data.task) {
                            throw new Error('Missing task payload');
                        }

                        syncTaskFromPayload(data.task);
                    } catch (error) {
                        placeCardIntoStatus(card, previousStatus);
                    }
                };

                getStatusColumns().forEach((column) => {
                    applyColumnColor(column, column.dataset.stageColor || '#94A3B8');

                    const moveHandle = column.querySelector('[data-stage-move-handle]');
                    if (moveHandle) {
                        moveHandle.addEventListener('dragstart', (event) => {
                            draggingStageColumn = column;
                            column.classList.add('opacity-70');

                            if (event.dataTransfer) {
                                event.dataTransfer.effectAllowed = 'move';
                                event.dataTransfer.setData('text/plain', column.dataset.statusColumn || 'stage-column');
                            }
                        });

                        moveHandle.addEventListener('dragend', () => {
                            column.classList.remove('opacity-70');
                            getStatusColumns().forEach((item) => {
                                item.classList.remove('ring-2', 'ring-indigo-200', 'ring-sky-200');
                            });
                            draggingStageColumn = null;
                        });
                    }

                    const colorPicker = column.querySelector('[data-stage-color-picker]');
                    if (colorPicker instanceof HTMLInputElement) {
                        colorPicker.addEventListener('input', () => {
                            applyColumnColor(column, colorPicker.value);
                            syncSettingsRowsFromBoard();
                        });

                        colorPicker.addEventListener('change', () => {
                            applyColumnColor(column, colorPicker.value);
                            syncSettingsRowsFromBoard();
                            queueStagePresetSave();
                        });
                    }
                });

                board.querySelectorAll('[data-task-card]').forEach((card) => {
                    card.addEventListener('dragstart', () => {
                        draggingCard = card;
                        card.classList.add('opacity-60');
                    });

                    card.addEventListener('dragend', () => {
                        card.classList.remove('opacity-60');
                        draggingCard = null;
                    });

                    const select = card.querySelector('[data-status-select]');
                    if (select) {
                        select.addEventListener('change', () => {
                            const nextStatus = select.value;
                            const prevStatus = card.dataset.taskStatus;
                            placeCardIntoStatus(card, nextStatus);
                            updateTaskStatus(card, nextStatus, prevStatus);
                        });
                    }
                });

                getStatusColumns().forEach((column) => {
                    column.addEventListener('dragover', (event) => {
                        if (!draggingCard && !draggingStageColumn) {
                            return;
                        }

                        event.preventDefault();
                        column.classList.add('ring-2', draggingStageColumn ? 'ring-sky-200' : 'ring-indigo-200');
                    });

                    column.addEventListener('dragleave', () => {
                        column.classList.remove('ring-2', 'ring-indigo-200', 'ring-sky-200');
                    });

                    column.addEventListener('drop', (event) => {
                        event.preventDefault();
                        column.classList.remove('ring-2', 'ring-indigo-200', 'ring-sky-200');

                        if (draggingStageColumn) {
                            const targetColumn = column;
                            if (targetColumn === draggingStageColumn) {
                                return;
                            }

                            const rect = targetColumn.getBoundingClientRect();
                            const shouldInsertAfter = event.clientX > rect.left + (rect.width / 2);
                            if (shouldInsertAfter) {
                                boardGrid.insertBefore(draggingStageColumn, targetColumn.nextElementSibling);
                            } else {
                                boardGrid.insertBefore(draggingStageColumn, targetColumn);
                            }

                            syncSettingsRowsFromBoard();
                            queueStagePresetSave();
                            return;
                        }

                        if (draggingCard) {
                            const targetStatus = column.dataset.statusColumn;
                            const currentStatus = draggingCard.dataset.taskStatus;
                            if (targetStatus === currentStatus) {
                                return;
                            }

                            placeCardIntoStatus(draggingCard, targetStatus);
                            updateTaskStatus(draggingCard, targetStatus, currentStatus);
                        }
                    });
                });

                if (window.Echo) {
                    window.Echo.private('tasks.board')
                        .listen('.task.status-changed', (payload) => {
                            if (payload.task) {
                                syncTaskFromPayload(payload.task);
                            }
                        });
                }

                updateAllStats();
            });
        </script>
    @endif
</x-app-layout>
