<?php

namespace App\Http\Controllers;

use App\Events\TaskStatusChanged;
use App\Models\Company;
use App\Models\Contact;
use App\Models\Deal;
use App\Models\Project;
use App\Models\ProjectStage;
use App\Models\Task;
use App\Models\User;
use App\Support\CrmModuleManager;
use App\Support\SectionAccessManager;
use App\Support\Tasks\TaskBoardBuilder;
use App\Support\Tasks\TaskFilterApplier;
use App\Support\Tasks\TaskKanbanStageManager;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class TaskController extends Controller
{
    public function __construct(
        private readonly TaskFilterApplier $taskFilterApplier,
        private readonly TaskBoardBuilder $taskBoardBuilder,
        private readonly TaskKanbanStageManager $taskKanbanStageManager
    )
    {
        $this->authorizeResource(Task::class, 'task');
    }

    public function index(Request $request, SectionAccessManager $sectionAccessManager): View
    {
        /** @var User $user */
        $user = $request->user();

        $kanbanStages = $this->taskKanbanStageManager->resolveKanbanStages($user);
        $statusOptions = collect($kanbanStages)
            ->mapWithKeys(fn (array $stage): array => [$stage['status'] => $stage['label']])
            ->all();

        $priorityOptions = [
            'low' => __('Low'),
            'medium' => __('Medium'),
            'high' => __('High'),
            'urgent' => __('Urgent'),
        ];

        $viewOptions = [
            'list' => __('List'),
            'kanban' => __('Kanban'),
            'gantt' => __('Gantt'),
        ];

        $parentModeOptions = [
            'all' => 'All',
            'parents' => 'Only the main ones',
            'subtasks' => 'Only subtasks',
            'with_subtasks' => 'With subtasks',
            'without_subtasks' => 'Without subtasks',
        ];

        $preferredViewMode = (string) ($user->preferred_task_view ?? 'list');
        if (! array_key_exists($preferredViewMode, $viewOptions)) {
            $preferredViewMode = 'list';
        }

        if ($request->has('view')) {
            $requestedViewMode = (string) $request->input('view', '');
            if (array_key_exists($requestedViewMode, $viewOptions)) {
                $viewMode = $requestedViewMode;

                if ((string) $user->preferred_task_view !== $viewMode) {
                    $user->forceFill([
                        'preferred_task_view' => $viewMode,
                    ])->saveQuietly();
                }
            } else {
                $viewMode = 'list';
            }
        } else {
            $viewMode = $preferredViewMode;
        }

        $search = trim((string) $request->input('q', ''));
        $status = $request->input('status');
        $priority = $request->input('priority');
        $assigneeId = $request->integer('assignee_id');
        $creatorId = $request->integer('creator_id');
        $projectId = $request->integer('project_id');
        $deadline = $request->input('deadline');
        $parentMode = (string) $request->input('parent_mode', 'all');
        if (! array_key_exists($parentMode, $parentModeOptions)) {
            $parentMode = 'all';
        }

        $dateFrom = trim((string) $request->input('date_from', ''));
        $dateTo = trim((string) $request->input('date_to', ''));
        $mine = $request->boolean('mine');

        $baseQuery = Task::query()
            ->with(['assignee', 'creator', 'deal', 'company', 'contact', 'project', 'projectStage', 'parent'])
            ->withCount(['subtasks', 'doneSubtasks']);

        $this->taskFilterApplier->apply($baseQuery, [
            'search' => $search,
            'status' => $status,
            'priority' => $priority,
            'assignee_id' => $assigneeId,
            'creator_id' => $creatorId,
            'project_id' => $projectId,
            'deadline' => $deadline,
            'parent_mode' => $parentMode,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
            'mine' => $mine,
            'user_id' => $user->id,
        ]);

        $tasks = null;
        $kanbanColumns = collect();
        $ganttRows = collect();
        $ganttPeriod = null;

        if ($viewMode === 'kanban') {
            $kanbanTasks = (clone $baseQuery)
                ->orderByRaw('due_at is null')
                ->orderBy('due_at')
                ->orderBy('sort_order')
                ->limit(400)
                ->get();

            $kanbanColumns = $this->taskBoardBuilder->buildKanbanColumns($kanbanTasks, $kanbanStages);
        } elseif ($viewMode === 'gantt') {
            $ganttTasks = (clone $baseQuery)
                ->orderByRaw('coalesce(starts_at, created_at)')
                ->orderByRaw('due_at is null')
                ->orderBy('due_at')
                ->limit(400)
                ->get();

            ['rows' => $ganttRows, 'period' => $ganttPeriod] = $this->taskBoardBuilder->buildGanttRows($ganttTasks);
        } else {
            $tasks = (clone $baseQuery)
                ->orderByRaw('due_at is null')
                ->orderBy('due_at')
                ->orderBy('sort_order')
                ->latest('id')
                ->paginate(25)
                ->withQueryString();
        }

        $users = User::query()->orderBy('name')->get();
        $projects = Project::query()->orderBy('name')->get();
        $canManageSectionAccess = $sectionAccessManager->canManage($user, 'tasks');
        $sectionAccessUsers = $canManageSectionAccess
            ? User::query()->orderBy('name')->get(['id', 'name', 'email', 'role', 'permissions'])
            : collect();

        return view('tasks.index', compact(
            'tasks',
            'users',
            'projects',
            'canManageSectionAccess',
            'sectionAccessUsers',
            'statusOptions',
            'priorityOptions',
            'viewOptions',
            'parentModeOptions',
            'kanbanStages',
            'kanbanColumns',
            'ganttRows',
            'ganttPeriod',
            'viewMode',
            'search',
            'status',
            'priority',
            'assigneeId',
            'creatorId',
            'projectId',
            'deadline',
            'parentMode',
            'dateFrom',
            'dateTo',
            'mine'
        ));
    }

    public function updateKanbanPreset(Request $request): RedirectResponse|JsonResponse
    {
        $this->authorize('viewAny', Task::class);

        $preset = $request->has('stages')
            ? $this->taskKanbanStageManager->buildPresetFromStagePayload($request)
            : $this->taskKanbanStageManager->buildPresetFromLegacyPayload($request);

        $request->user()->forceFill([
            'task_kanban_preset' => $preset,
        ])->save();

        if ($request->expectsJson()) {
            return response()->json([
                'message' => 'Kanban of tasks saved.',
                'stages' => $preset,
            ]);
        }

        return back()->with('success', 'Kanban of tasks saved.');
    }

    public function create(Request $request): View
    {
        return view('tasks.create', [
            ...$this->formData($request->user()),
            'requestedProjectId' => $request->integer('project_id'),
            'requestedProjectStageId' => $request->integer('project_stage_id'),
            'requestedParentId' => $request->integer('parent_id'),
        ]);
    }

    public function store(Request $request, CrmModuleManager $moduleManager): RedirectResponse
    {
        $payload = $this->validatedData($request);
        $payload['creator_id'] = $request->user()->id;
        $payload = $moduleManager->applyPayloadHooks('tasks.store', $payload, [
            'hook' => 'tasks.store',
            'user_id' => $request->user()->id,
        ], array_keys($payload));
        $payload['completed_at'] = ($payload['status'] ?? null) === 'done'
            ? ($payload['completed_at'] ?? now())
            : null;

        $task = Task::create($payload);

        $this->refreshProjectProgress($task);
        $this->syncParentStatusById($task->parent_id);

        return redirect()
            ->route('tasks.show', $task)
            ->with('success', 'The task has been created.');
    }

    public function show(Request $request, Task $task): View
    {
        $task->load([
            'creator',
            'assignee',
            'deal',
            'company',
            'contact',
            'project',
            'projectStage',
            'parent',
            'subtasks.assignee',
            'subtasks.creator',
            'subtasks.project',
        ]);

        $subtasks = $task->subtasks
            ->sortBy([
                ['status', 'asc'],
                ['due_at', 'asc'],
                ['id', 'asc'],
            ])
            ->values();

        $subtaskStats = [
            'total' => $subtasks->count(),
            'done' => $subtasks->where('status', 'done')->count(),
        ];
        $subtaskStats['progress'] = $subtaskStats['total'] > 0
            ? (int) round(($subtaskStats['done'] / $subtaskStats['total']) * 100)
            : 0;

        $formData = $this->formData($request->user(), $task);
        $viewPayload = [
            ...$formData,
            'task' => $task,
            'subtasks' => $subtasks,
            'subtaskStats' => $subtaskStats,
        ];

        if ($request->boolean('sidepanel') || $request->header('X-Sidepanel') === '1') {
            return view('sidepanel.tasks.show', $viewPayload);
        }

        return view('tasks.show', $viewPayload);
    }

    public function edit(Task $task): View
    {
        /** @var User $user */
        $user = auth()->user();

        return view('tasks.edit', [
            ...$this->formData($user, $task),
            'task' => $task,
            'requestedProjectId' => null,
            'requestedProjectStageId' => null,
            'requestedParentId' => null,
        ]);
    }

    public function update(Request $request, Task $task, CrmModuleManager $moduleManager): RedirectResponse|JsonResponse
    {
        $previousProjectId = $task->project_id;
        $previousParentId = $task->parent_id;

        $payload = $this->validatedData($request, $task);
        $payload = $moduleManager->applyPayloadHooks('tasks.update', $payload, [
            'hook' => 'tasks.update',
            'user_id' => $request->user()->id,
            'task_id' => $task->id,
        ], array_keys($payload));
        $payload['completed_at'] = ($payload['status'] ?? null) === 'done'
            ? ($payload['completed_at'] ?? now())
            : null;

        $task->update($payload);
        $task->refresh()->load(['assignee', 'creator', 'parent', 'project']);
        $task->loadCount(['subtasks', 'doneSubtasks']);

        $this->refreshProjectProgress($task);
        if ($previousProjectId && $previousProjectId !== $task->project_id) {
            Project::query()->find($previousProjectId)?->recalculateProgress();
        }

        if ($previousParentId !== $task->parent_id) {
            $this->syncParentStatusById($previousParentId);
        }
        $this->syncParentStatusById($task->parent_id);

        if ($request->expectsJson()) {
            return response()->json([
                'message' => 'The task has been updated.',
                'task' => $this->taskJsonPayload($task),
            ]);
        }

        return redirect()
            ->route('tasks.show', $task)
            ->with('success', 'The task has been updated.');
    }

    public function destroy(Task $task): RedirectResponse
    {
        $projectId = $task->project_id;
        $parentId = $task->parent_id;

        $task->delete();

        if ($projectId) {
            Project::query()->find($projectId)?->recalculateProgress();
        }

        $this->syncParentStatusById($parentId);

        return redirect()
            ->route('tasks.index')
            ->with('success', 'The task has been deleted.');
    }

    public function complete(Task $task): RedirectResponse
    {
        $this->authorize('complete', $task);

        $doneStageId = null;

        if ($task->project_id) {
            $doneStageId = ProjectStage::query()
                ->where('project_id', $task->project_id)
                ->where('is_done', true)
                ->value('id');
        }

        $task->update([
            'status' => 'done',
            'project_stage_id' => $doneStageId ?? $task->project_stage_id,
            'completed_at' => now(),
        ]);

        $task->refresh()->load(['assignee', 'creator', 'parent']);

        $this->refreshProjectProgress($task);
        $this->syncParentStatusById($task->parent_id);

        $this->broadcastToOthers(new TaskStatusChanged($task));

        return back()->with('success', 'The task is marked as completed.');
    }

    public function updateStatus(Request $request, Task $task): RedirectResponse|JsonResponse
    {
        $this->authorize('update', $task);

        $validated = $request->validate([
            'status' => ['required', Rule::in($this->taskKanbanStageManager->allowedTaskStatusesForUser($request->user(), $task->status))],
        ]);

        $status = (string) $validated['status'];
        $projectStageId = $task->project_stage_id;

        if ($task->project_id) {
            if ($status === 'done') {
                $doneStageId = ProjectStage::query()
                    ->where('project_id', $task->project_id)
                    ->where('is_done', true)
                    ->value('id');

                if ($doneStageId) {
                    $projectStageId = $doneStageId;
                }
            } elseif ($task->projectStage?->is_done) {
                $projectStageId = ProjectStage::query()
                    ->where('project_id', $task->project_id)
                    ->where('is_done', false)
                    ->orderBy('sort_order')
                    ->value('id');
            }
        }

        $task->update([
            'status' => $status,
            'project_stage_id' => $projectStageId,
            'completed_at' => $status === 'done' ? ($task->completed_at ?? now()) : null,
        ]);

        $task->refresh()->load(['assignee', 'creator', 'parent']);
        $task->loadCount(['subtasks', 'doneSubtasks']);

        $this->refreshProjectProgress($task);
        $this->syncParentStatusById($task->parent_id);

        $this->broadcastToOthers(new TaskStatusChanged($task));

        if ($request->expectsJson()) {
            return response()->json([
                'message' => 'The task status has been updated.',
                'task' => $this->taskJsonPayload($task),
            ]);
        }

        return back()->with('success', 'The task status has been updated.');
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedData(Request $request, ?Task $task = null): array
    {
        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'deal_id' => ['nullable', 'exists:deals,id'],
            'company_id' => ['nullable', 'exists:companies,id'],
            'contact_id' => ['nullable', 'exists:contacts,id'],
            'project_id' => ['nullable', 'exists:projects,id'],
            'project_stage_id' => ['nullable', 'exists:project_stages,id'],
            'parent_id' => ['nullable', 'exists:tasks,id'],
            'assignee_id' => ['nullable', 'exists:users,id'],
            'status' => ['required', Rule::in($this->taskKanbanStageManager->allowedTaskStatusesForUser($request->user(), $task?->status))],
            'priority' => ['required', Rule::in(['low', 'medium', 'high', 'urgent'])],
            'estimated_hours' => ['nullable', 'numeric', 'min:0'],
            'tracked_hours' => ['nullable', 'numeric', 'min:0'],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'starts_at' => ['nullable', 'date'],
            'due_at' => ['nullable', 'date', 'after_or_equal:starts_at'],
            'reminder_at' => ['nullable', 'date'],
        ]);

        $parentId = $validated['parent_id'] ?? null;
        $projectId = $validated['project_id'] ?? null;
        $projectStageId = $validated['project_stage_id'] ?? null;

        if ($parentId) {
            $parent = Task::query()->findOrFail((int) $parentId);

            if ($task && $parent->id === $task->id) {
                throw ValidationException::withMessages([
                    'parent_id' => 'A task cannot be a subtask of itself.',
                ]);
            }

            if ($task && $this->isDescendantOfTask($parent, $task->id)) {
                throw ValidationException::withMessages([
                    'parent_id' => 'You cannot select a child task as a parent task.',
                ]);
            }

            if (! $projectId && $parent->project_id) {
                $projectId = $parent->project_id;
                $validated['project_id'] = $projectId;
            }

            if ($parent->project_id && (int) $projectId !== (int) $parent->project_id) {
                throw ValidationException::withMessages([
                    'project_id' => 'The subtask must belong to the same project as the parent task.',
                ]);
            }
        }

        if ($projectStageId && ! $projectId) {
            throw ValidationException::withMessages([
                'project_stage_id' => 'To select a stage, first select a project.',
            ]);
        }

        if ($projectId && $projectStageId) {
            $stage = ProjectStage::query()->findOrFail((int) $projectStageId);

            if ($stage->project_id !== (int) $projectId) {
                throw ValidationException::withMessages([
                    'project_stage_id' => 'The stage does not belong to the selected project.',
                ]);
            }

            if ($stage->is_done) {
                $validated['status'] = 'done';
            }
        }

        if (! $projectId) {
            $validated['project_stage_id'] = null;
        }

        $validated['project_id'] = $projectId;
        $validated['estimated_hours'] = (float) ($validated['estimated_hours'] ?? 0);
        $validated['tracked_hours'] = (float) ($validated['tracked_hours'] ?? 0);
        $validated['sort_order'] = (int) ($validated['sort_order'] ?? 0);
        $validated['completed_at'] = $validated['status'] === 'done'
            ? ($task?->completed_at ?? now())
            : null;

        return $validated;
    }

    /**
     * @return array<string, mixed>
     */
    private function formData(User $user, ?Task $task = null): array
    {
        $deals = Deal::query()->latest()->limit(200)->get();
        $companies = Company::query()->orderBy('name')->get();
        $contacts = Contact::query()->orderBy('first_name')->orderBy('last_name')->get();
        $users = User::query()->orderBy('name')->get();
        $projects = Project::query()
            ->with(['stages' => fn ($query) => $query->orderBy('sort_order')])
            ->orderBy('name')
            ->get();
        $parentTasks = Task::query()->orderByDesc('id')->limit(300)->get(['id', 'title', 'status']);
        $taskStatusOptions = collect($this->taskKanbanStageManager->resolveKanbanStages($user))
            ->mapWithKeys(fn (array $stage): array => [$stage['status'] => $stage['label']])
            ->all();

        if ($task && ! array_key_exists($task->status, $taskStatusOptions)) {
            $taskStatusOptions[$task->status] = Str::headline(str_replace('_', ' ', $task->status));
        }

        return compact('deals', 'companies', 'contacts', 'users', 'projects', 'parentTasks', 'taskStatusOptions');
    }

    /**
     * @return array<string, mixed>
     */
    private function taskJsonPayload(Task $task): array
    {
        return [
            'id' => $task->id,
            'title' => $task->title,
            'status' => $task->status,
            'priority' => $task->priority,
            'parent_id' => $task->parent_id,
            'parent_title' => $task->parent?->title,
            'project_id' => $task->project_id,
            'project_name' => $task->project?->name,
            'assignee_name' => $task->assignee?->name,
            'creator_name' => $task->creator?->name,
            'starts_at' => $task->starts_at?->toIso8601String(),
            'due_at' => $task->due_at?->toIso8601String(),
            'tracked_hours' => (float) $task->tracked_hours,
            'estimated_hours' => (float) $task->estimated_hours,
            'subtasks_count' => $task->subtasks_count,
            'done_subtasks_count' => $task->done_subtasks_count,
            'url' => route('tasks.show', $task),
            'edit_url' => route('tasks.show', $task),
        ];
    }

    private function refreshProjectProgress(Task $task): void
    {
        if ($task->project_id) {
            $task->project?->recalculateProgress();
        }
    }

    private function syncParentStatusById(?int $parentId): void
    {
        if (! $parentId) {
            return;
        }

        $parent = Task::query()->find($parentId);

        if (! $parent) {
            return;
        }

        $stats = $parent->subtasks()
            ->selectRaw('count(*) as total')
            ->selectRaw("sum(case when status = 'done' then 1 else 0 end) as done")
            ->first();

        $total = (int) ($stats?->total ?? 0);

        if ($total === 0) {
            return;
        }

        $done = (int) ($stats?->done ?? 0);

        if ($done === $total && $parent->status !== 'done') {
            $parent->update([
                'status' => 'done',
                'completed_at' => $parent->completed_at ?? now(),
            ]);

            $this->refreshProjectProgress($parent);
            $this->broadcastToOthers(new TaskStatusChanged($parent));

            return;
        }

        if ($done < $total && $parent->status === 'done') {
            $parent->update([
                'status' => 'in_progress',
                'completed_at' => null,
            ]);

            $this->refreshProjectProgress($parent);
            $this->broadcastToOthers(new TaskStatusChanged($parent));
        }
    }

    private function isDescendantOfTask(Task $candidateParent, int $taskId): bool
    {
        $visited = [];
        $current = $candidateParent;

        while ($current) {
            if ($current->id === $taskId) {
                return true;
            }

            if (isset($visited[$current->id]) || ! $current->parent_id) {
                return false;
            }

            $visited[$current->id] = true;
            $current = Task::query()->find($current->parent_id);
        }

        return false;
    }

}
