<?php

namespace App\Http\Controllers;

use App\Models\UserMenuItem;
use App\Support\MenuManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Validation\Rule;

class ProfileMenuController extends Controller
{
    public function update(Request $request, MenuManager $menuManager): RedirectResponse|Response
    {
        $user = $request->user();
        $menuManager->ensureCoreItems($user);

        $validated = $request->validate([
            'order' => ['required', 'array', 'min:1'],
            'order.*' => [
                'required',
                'integer',
                Rule::exists('user_menu_items', 'id')->where(fn ($query) => $query->where('user_id', $user->id)),
            ],
            'visible' => ['nullable', 'array'],
            'visible.*' => ['nullable', 'boolean'],
            'custom' => ['nullable', 'array'],
            'custom.*.label' => ['nullable', 'string', 'max:120'],
            'custom.*.url' => ['nullable', 'url', 'max:2048'],
            'custom.*.icon' => ['nullable', Rule::in(MenuManager::iconNames())],
            'custom.*.open_in_new_tab' => ['nullable', 'boolean'],
            'custom.*.delete' => ['nullable', 'boolean'],
        ]);

        $order = collect($validated['order'])
            ->map(fn ($id) => (int) $id)
            ->values();

        $items = UserMenuItem::query()
            ->where('user_id', $user->id)
            ->whereIn('id', $order)
            ->get()
            ->keyBy('id');

        foreach ($order as $position => $itemId) {
            $item = $items->get($itemId);
            if (! $item) {
                continue;
            }

            $item->sort_order = $position;
            $item->is_visible = array_key_exists((string) $itemId, $validated['visible'] ?? []);

            if ($item->is_custom) {
                $payload = $validated['custom'][(string) $itemId] ?? $validated['custom'][$itemId] ?? null;
                if (is_array($payload)) {
                    $shouldDelete = ! empty($payload['delete']);
                    if ($shouldDelete) {
                        $item->delete();

                        continue;
                    }

                    $label = trim((string) ($payload['label'] ?? $item->label));
                    $url = trim((string) ($payload['url'] ?? $item->url));
                    $icon = (string) ($payload['icon'] ?? $item->icon);
                    $openInNewTab = ! empty($payload['open_in_new_tab']);

                    if ($label !== '' && $url !== '') {
                        $item->label = $label;
                        $item->url = $url;
                        $item->icon = $icon;
                        $item->open_in_new_tab = $openInNewTab;
                    }
                }
            }

            $item->save();
        }

        $menuManager->forgetUserCaches($user->id);

        if ($request->expectsJson()) {
            return response()->noContent();
        }

        return back()->with('success', 'The user menu has been updated.');
    }

    public function storeCustom(Request $request, MenuManager $menuManager): RedirectResponse
    {
        $validated = $request->validate([
            'label' => ['required', 'string', 'max:120'],
            'url' => ['required', 'url', 'max:2048'],
            'icon' => ['required', Rule::in(MenuManager::iconNames())],
            'open_in_new_tab' => ['nullable', 'boolean'],
        ]);

        $user = $request->user();
        $menuManager->ensureCoreItems($user);

        $nextSortOrder = (int) $user->menuItems()->max('sort_order') + 1;

        $user->menuItems()->create([
            'key' => null,
            'label' => trim((string) $validated['label']),
            'url' => trim((string) $validated['url']),
            'icon' => (string) $validated['icon'],
            'sort_order' => $nextSortOrder,
            'is_visible' => true,
            'is_custom' => true,
            'open_in_new_tab' => $request->boolean('open_in_new_tab'),
        ]);

        $menuManager->forgetUserCaches($user->id);

        return back()->with('success', 'Custom menu item added.');
    }

    public function updateSidebar(Request $request): RedirectResponse|Response
    {
        $validated = $request->validate([
            'collapsed' => ['required', 'boolean'],
        ]);

        $request->user()->update([
            'sidebar_collapsed' => (bool) $validated['collapsed'],
        ]);

        if ($request->expectsJson()) {
            return response()->noContent();
        }

        return back();
    }

    public function updateRightRail(Request $request): RedirectResponse|Response
    {
        $validated = $request->validate([
            'collapsed' => ['required', 'boolean'],
        ]);

        $request->user()->update([
            'right_rail_collapsed' => (bool) $validated['collapsed'],
        ]);

        if ($request->expectsJson()) {
            return response()->noContent();
        }

        return back();
    }
}
